import unittest
import sys
import os

from signalling_test import Signalling_Test
from gasdata_file import GasData_File

#got the following to work by setting the environment variable PYTHONPATH=C:\Program Files\QGIS 3.10\apps\qgis-ltr\python
##import qgis.core  # path test

# but *this* needed cleverness (see our wrapping batch files)
##from qgis.PyQt.QtCore import *


GASFILE = os.path.join("testdata","gis.small.gas")
GASFILE_TEXT_IN_NUM = os.path.join("testdata","GIS.SMALL.text_in_tin.gas")
GASFILE_UA = os.path.join("testdata","gis.small.ua.gas")
GASFILE_NONEARTH = os.path.join("testdata","gis.small.nonearth.gas")
GASFILE_63 = os.path.join("testdata","gis.small.6.3.gas")
GASFILE_61 = os.path.join("testdata","gis.small.6.1.gas")
GASFILE_NOSEL = os.path.join("testdata","GIS.SMALL.NOSEL.HIST.7.3.gas") #bug #4482 IS in this file saved with gas 7.3
GASFILE_NONORTH = os.path.join("testdata","gis.small.no.north.gas")
GASFILE_DUPEAST = os.path.join("testdata","gis.small.dupe.east.gas")
GASFILE_EN_DUP = os.path.join("testdata","GIS.SMALL.EAST.NORTH.DUPLICATED.gas")
GASFILE_LL = os.path.join("testdata","GIS.SMALL.LONG.LAT.gas")
GASFILE_EN_AND_L = os.path.join("testdata","GIS.SMALL.ENL.gas")
GASFILE_OLD_EXPORT = os.path.join("testdata","GIS.SMALL.exported.from.7.1.gas")
GASFILE_DOWNHOLE = os.path.join("testdata","GIS.DOWNHOLE.gas")
UNICODE = os.path.join("testdata","unicode.ansi.gas")

# module level as can't figure out how to pass properties into the class (for silentmode)
_signalling = Signalling_Test() # placeholder


class Test_GasFileLoading(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        # previously i had the signalling static, but this is neater
        _signalling.log(0,'setUpClass',f'>>> SETUP of test class  {str(cls)}')


    def test_atts(self):
        testName='test_atts'
        print(f"\n>>> {testName}")

        g = GasData_File(GASFILE,  _signalling)
        g.load_data()


        #colours
        self.assertEqual(6, len(g.colours)) #6  colours in test file
        self.assertEqual("Default Colour", g.colours[0].name)
        self.assertEqual("A", g.colours[1].name) #all properties of first one
        self.assertEqual("-65536", g.colours[1].colour) #red
        self.assertTrue(g.colours[1].visible)

        self.assertEqual("-16711936", g.colours[2].colour) #2nd green

        self.assertFalse(g.colours[4].visible)
        self.assertEqual("INVIZ", g.colours[4].name)
        self.assertEqual("D", g.colours[5].name)

        # shapes
        self.assertEqual(3, len(g.shapes))
        self.assertEqual("aa", g.shapes[1].name)
        self.assertEqual("1", g.shapes[1].shapeCode)
        self.assertTrue(g.shapes[1].visible)
        self.assertEqual("2", g.shapes[2].shapeCode)

        # sizes
        self.assertEqual(4, len(g.sizes))
        self.assertEqual("_x", g.sizes[1].name)
        self.assertEqual("8", g.sizes[1].size)
        self.assertTrue(g.sizes[1].visible)
        self.assertEqual("10", g.sizes[2].size)

        # titles
        self.assertEqual("atta, attb, _attc", g.symbologyTitle)

        #unique att combinations - sorted
        self.assertEqual(7, len(g.uniqueAttCombinations))
        self.assertEqual((1,1,1), (g.uniqueAttCombinations[0]))
        self.assertEqual((1,1,2), (g.uniqueAttCombinations[1]))
        self.assertEqual((2,2,2), (g.uniqueAttCombinations[2]))


    def test_uniqueAtts(self):
        testName='test_uniqueAtts'
        print(f"\n>>> {testName}")
        g = GasData_File(GASFILE_UA,  _signalling)
        g.load_data()


        #colours
        self.assertEqual(3, len(g.colours)) #6  colours in test file
        self.assertEqual("Default Colour", g.colours[0].name)
        

        #unique att combinations - sorted
        self.assertEqual(4, len(g.uniqueAttCombinations)) # was 6 prior to bugfix #4390
        self.assertEqual((1,1,0), (g.uniqueAttCombinations[0]))


    def test_data(self):
        testName='test_data'
        print(f"\n>>> {testName}")
        g = GasData_File(GASFILE, _signalling)
        g.load_data()

        d = g.data
        a = g.atts
        da = g.dataAndAtts

        #special and selected column names
        #todo put in test below: self.assertEqual(('id', 'east', 'north', 'RL', 'copper', 'tin', 'lead', 'code'),  g.columnNames)

        #column names and types
        self.assertEqual("id", str(g.columns[0].name))
        self.assertEqual("Numeric", str(g.columns[0].type))
        self.assertEqual("copper_", str(g.columns[4].name)) #deliberate _ in name to test escaping
        self.assertEqual("Numeric", str(g.columns[4].type))
        self.assertEqual("code", str(g.columns[7].name))
        self.assertEqual("Text", str(g.columns[7].type))

        #visbile row count
        self.assertEqual(9, len(d))

        #4 sel and 4 special cols
        self.assertEqual(8, len(d[0]))

        #4 att columns
        self.assertEqual(4, len(a[0]))

        #both
        self.assertEqual(4+8, len(da[0]))

        #data values for copper
        self.assertEqual(4.0, d[0][4])
        self.assertEqual(None, d[1][4]) #null

        #data values for code
        self.assertEqual("hello", d[0][7])
        self.assertEqual("there_1", d[1][7])
        self.assertEqual("", d[2][7]) #null

        #test colour of 4th row
        colourAtt = int(a[3][1])
        colour = g.colours[colourAtt]
        self.assertEqual("D", colour.name)
        self.assertEqual("-16776961", colour.colour) #blue

    def test_textInNumeric(self):
        testName='test_textInNumeric'
        print(f"\n>>> {testName}")
        g = GasData_File(GASFILE_TEXT_IN_NUM, _signalling)
        g.load_data()

        d = g.data
        a = g.atts
        da = g.dataAndAtts

        #special and selected column names
        #todo put in test below: self.assertEqual(('id', 'east', 'north', 'RL', 'copper', 'tin', 'lead', 'code'),  g.columnNames)

        #column names and types
        
        self.assertEqual("copper_", str(g.columns[5].name)) 
        self.assertEqual("Numeric", str(g.columns[5].type))
       

        #visbile row count
        self.assertEqual(10, len(d))
    

        #data values for tin
        self.assertEqual(None, d[0][6])
        self.assertEqual(None, d[1][6]) # x becomes none as its not numeric
        self.assertEqual(6.0,  d[2][6])
        self.assertEqual(7.0,  d[3][6])

       


    def test_noSelection(self):
        testName='test_noSelection'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_NOSEL,  _signalling)
        g.load_data()

        d = g.data

        # visbile row count
        self.assertEqual(9, len(d))

        # 0 sel and 4 special cols  
        #print (f'DEBUG: {g.debugmsg}')
        #print (f'test_noSelection(columns) {str(g.columns)}')  
        self.assertEqual(4, len(d[0]))

    def test_noNorth(self):
        testName='test_noNorth'
        print(f"\n>>> {testName}")
        try:
            g = GasData_File(GASFILE_NONORTH,  _signalling)
            
            self.assertFalse(g.load_data()) # should not load data as north column missing
            
        except ValueError as e:
            self.assertEqual('East and North fields must be set in ioGAS', str(e))

    # extra east special col is ignored
    def test_dupEast(self):
        testName='test_dupEast'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_DUPEAST,  _signalling)
        g.load_data()

        d = g.data

        # column names and types
        self.assertEqual("id", str(g.columns[0].name))
        self.assertEqual("Numeric", str(g.columns[0].type))
        self.assertEqual("east", str(g.columns[1].name))
        self.assertEqual("Numeric", str(g.columns[1].type))
        self.assertEqual("copper_", str(g.columns[2].name))
        self.assertEqual("Numeric", str(g.columns[2].type))
        self.assertEqual("tin", str(g.columns[3].name))
        self.assertEqual("Numeric", str(g.columns[3].type))

        # 4 sel and 2 special cols (id and east)
        self.assertEqual(6, len(d[0]))

        # data values for copper
        self.assertEqual(4, d[0][2])
        self.assertEqual(None, d[1][2])  # null

    # setting LL to EN does nothing as EN are already used
    def test_EN_and_LL(self):
        testName='test_EN_and_LL'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_EN_DUP,  _signalling)
        g.load_data()

        d = g.data

        # column names and types
        self.assertEqual("id", str(g.columns[0].name))


        # 4 sel and 4 special cols - id/e/n/rl
        self.assertEqual(8, len(d[0]))

    # bring in total of 6 special cols, and zero selected
    def test_EN_and_L(self):
        testName='test_EN_and_L'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_EN_AND_L,  _signalling)

        g.load_data()

        d = g.data
        self.assertEqual(6, len(d[0]))

        # column names and types
        self.assertEqual("id", str(g.columns[0].name))
        self.assertEqual("east", str(g.columns[1].name))
        self.assertEqual("north", str(g.columns[2].name))
        self.assertEqual("tin", str(g.columns[3].name)) # not selected but set as LONGITUDE
        self.assertEqual("copper_", str(g.columns[4].name)) # not selected but set as date time
        self.assertEqual("RL", str(g.columns[5].name))


    #LL is not enough, you need E,N
    def test_LL(self):
        testName='test_LL'
        print(f"\n>>> {testName}")
        
        try:
            g = GasData_File(GASFILE_LL,  _signalling)

            # gas file loading should return false as this file has no map_east,north spec cols
            self.assertFalse(g.load_data())
            #    self.fail('File Should not have loaded as missing coord special cols')

            #d = g.data
            #a = g.atts
            #da = g.dataAndAtts

        # TODO - review this ex
        except ValueError as e:
            self.assertEqual('East and North fields must be set in ioGAS', str(e))

    def test_specialColNames(self):
        testName='test_specialColNames'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE,  _signalling)
        g.load_data()


        self.assertEqual("east", g.specialColumnEast)
        self.assertEqual("north", g.specialColumnNorth)
        self.assertEqual("32613", g.specialColumnEPSG)
        self.assertEqual("id", g.specialColumnID)

        self.assertEqual("id", g.columns[0].name)
        #east is *2nd* (id/e/n/elev are at start) despite being selected at the *end*
        self.assertEqual("east", g.columns[1].name)

    def test_nonEarth(self):
        testName='test_nonEarth'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_NONEARTH,  _signalling)
        g.load_data()

        self.assertEqual("east", g.specialColumnEast)
        self.assertEqual("north", g.specialColumnNorth)
        self.assertEqual("0", g.specialColumnEPSG)
        #can't assert proj='' as we don't have it in gasdata (its in messaging underneath)
        
    # weak test.  Just proves we can read unicode without crashing, but the char translation
    # is strange...
    def test_unicode(self):
        testName='test_unicode'
        print(f"\n>>> {testName}")
        
        g = GasData_File(UNICODE, _signalling)
        g.load_data()

        d = g.data
        a = g.atts
        da = g.dataAndAtts

        #visbile row count
        self.assertEqual(3, len(d))
        # print(f' d is {d} ')

        # read some data
        self.assertEqual(1.0, d[0][0]) 
        # skip  east and north cols
        self.assertEqual(2.0, d[0][3])
        self.assertEqual("blah", d[0][4]) 
           
        self.assertEqual("L", d[1][4][0:1]) 
        # can't get the assert to pass on 2nd char no matter what I paste in!
        # self.assertEqual"", d[1][4][1:2]) 
        # IT SHOULD BE LO WITH A TILDA ON THE O 

        # anyway its not crashing because I can read the first character of the cell below
        self.assertEqual("w", d[2][4][0:1]) 

    def test_downhole_specialCols(self):
        testName='test_downhole_specialCols'
        print(f"\n>>> {testName}")
        
        g = GasData_File(GASFILE_DOWNHOLE,  _signalling)
        g.load_data()

        self.assertEqual("east", str(g.columns[0].name))
        self.assertEqual("north", str(g.columns[1].name))
        self.assertEqual("RL", str(g.columns[2].name))
        self.assertEqual("hole", str(g.columns[3].name))
        self.assertEqual("from", str(g.columns[4].name))
        self.assertEqual("to", str(g.columns[5].name))
        self.assertEqual("lead", str(g.columns[6].name)) #selected


        self.assertEqual("east", g.specialColumnEast)
        self.assertEqual("north", g.specialColumnNorth)
        self.assertEqual("hole", g.specialColumnGroup)
        self.assertEqual("from", g.specialColumnFrom)
        self.assertEqual("to", g.specialColumnTo)


    def test_versioning(self):
        testName='test_versioning'
        print(f"\n>>> {testName}")

        try:
            #correct version number reading and loading of >=6.3  and any weird non-numerics
            g = GasData_File(GASFILE_63, _signalling)
            self.assertTrue(g.load_data())  # numeric version 6.3 should load
            self.assertEqual(6.3, g.version) 
            
            g = GasData_File(GASFILE, _signalling)
            self.assertTrue(g.load_data()) # weird non numeric dev version should load
            self.assertEqual(float(6.3), g.version) # version is coerced into 6.3
        except:
            self.fail()

        try:
            #can't load old versions
            g = GasData_File(GASFILE_61, _signalling)
            self.assertFalse(g.load_data())  # numeric <6.3 should fail to load
            self.assertEqual(6.1, g.version)
            #self.fail()

        except ValueError as e:
            self.assertEqual("File must have been saved with ioGAS version 6.3 or later", str(e))

        #warning on middle aged versions (see #4148)
        g = GasData_File(GASFILE_OLD_EXPORT, _signalling) # this file has no version file
        self.assertTrue(g.load_data())   # but we assume good and load with warning
        self.assertEqual(6.3, g.version) #blank version is coerced into 6.3



if __name__ == '__main__':

    # Get the command line arg we need
    
    # Control output noise in build reporting
    silentMode=int(sys.argv[1])
    print (f'silentmode set to {silentMode}')
  
	# Apparently neccassary to reconstruct minimal args :
    a = [1] # make a new array
    a[0] = sys.argv[0]
    sys.argv = a
    
    _signalling = Signalling_Test(testName = 'main',silentMode=int(silentMode))

    unittest.main()