# Main plugin entry point

import os
from pathlib import Path
import tempfile
import threading 
import configparser
import time
import asyncio
import threading
from datetime import datetime
import traceback

from qgis.core import * 
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMenu,QToolButton,QMessageBox

#start of this path 'ioGAS' must match deployment
from ioGAS.signalling_app import Signalling_App
from ioGAS.gasdata_file import GasData_File
from ioGAS.gasdata_link import GasData_Link
from ioGAS.gaslayerlogic_file import GasLayerLogic_File
from ioGAS.gaslayerlogic_link import GasLayerLogic_Link
from ioGAS.gaslinkclient_app import GasLinkClient_App
from ioGAS.gaslinkcomms import GasLinkComms
from ioGAS.gassupport import *

pluginPath = os.path.dirname(__file__)

buildID  = "202054"
majorVersion = "8.2"
verDev = "DEV:1.01"   # dev tracking only for when buildID is not being set by build

# Logging : Will only display levels in this this list; 
# suggested convention : 0=prod,user info; 1=Prod, user warning; 2=Dev-warning/info; 3=Dev-specific debug nsg
LOG_LEVELS = [0,1,2,3] # = [0] or  [0,1] for production, = [3] to just get a specific debug message

LIMIT = 60  # max number of unique attributes we can handle in Lnear legend mode on qgis < 3.3 (sqllite limitation)

vectorMenuText ='Import ioGAS File'


class QGisPluginGas:
    #This plugin class is initialised when plugin is loaded? 
    def __init__(self, iface):
        self.iface = iface
        
        # config is stored in ini file in user roaming appdata
        self.config, configMsg = read_or_initialise_config()
        self.logFilePath = get_logfile_path(True)
   
        # set up signalling/logging
        self.signalling = Signalling_App()
        self.signalling.sig_log_raised.connect(self.on_log_raised)     # wire up logging to  UI
        self.signalling.sig_notify_raised.connect(self.on_notify_raised)     #  wire up notifications to UI
        self.signalling.log(0,'QGisPluginGas.__init__()', f'Initialising live link plugin [Build:{buildID if "##" not in buildID else verDev},majorVersion:{majorVersion}\nConfig:{configMsg}\nLogging to:{self.logFilePath}]')

       
        # create our GLM Client app for live link
        # NB we must wire up the glmClient user_cancelled signal beore wiring it up to this plugin - we need the glmclient user_cancelled handler to be called first.
        self.glmClient = GasLinkClient_App(self.signalling,self.config,self.iface)
        
        self.signalling.sig_connected.connect(self.on_ToggleLivelinkConnect)  #wire up this level's handlers for notification of connected/disconnected
        self.signalling.sig_connection_lost.connect(self.on_ToggleLivelinkConnect)
        self.signalling.sig_listener_exception.connect(self.on_ToggleLivelinkConnect)
        self.signalling.sig_user_cancelled.connect(self.on_ToggleLivelinkConnect)  
        

        
        # gas file loading params
        self.legendType = 'LINEAR' # Hierachical / Linear; currently we are only supporting Linear
        self.layerType = '' # Scratch / Geopackage
 
    # set this plugin up in QGIS UI : toolbar dropdown and vector menu submenu
    def initGui(self):

        # Create 2 actions - for scracth and geopackage

        self.action_GasFileScratch = QAction('Scratch Layer', self.iface.mainWindow())
        self.action_GasFileScratch.setIcon(QIcon(os.path.join(pluginPath, 'icons', 'iogas.svg')))
        self.action_GasFileScratch.triggered.connect(self.runScratch)

        self.action_GasFileGeopkg = QAction('Create GeoPackage', self.iface.mainWindow())
        self.action_GasFileGeopkg.setIcon(QIcon(os.path.join(pluginPath, 'icons', 'iogas.svg')))
        self.action_GasFileGeopkg.triggered.connect(self.runLinear)

        self.action_LiveLink = QAction('Live Link', self.iface.mainWindow())
        self.action_LiveLink.setIcon(QIcon(os.path.join(pluginPath, 'icons', 'iogas.svg')))
        self.action_LiveLink.triggered.connect(self.connectLiveLink)

        # Add the actions as submenu items to the Vector menu
        self.iface.addPluginToVectorMenu(vectorMenuText, self.action_GasFileScratch)   # submenu name, menuItem
        self.iface.addPluginToVectorMenu(vectorMenuText, self.action_GasFileGeopkg)
        self.iface.addPluginToVectorMenu(vectorMenuText, self.action_LiveLink)

        # For the toolbar widget we create  a popup menu triggered from a toolbutton
        self.popupMenu = QMenu(self.iface.mainWindow())
        self.popupMenu.addAction(self.action_GasFileScratch)
        self.popupMenu.addAction(self.action_GasFileGeopkg)
        self.popupMenu.addAction(self.action_LiveLink)
  
        self.toolButton = QToolButton()
        self.toolButton.setIcon(QIcon(os.path.join(pluginPath, 'icons', 'iogas.svg')))
        self.toolButton.setToolTip(vectorMenuText)
        self.toolButton.setMenu(self.popupMenu)
        self.toolButton.setPopupMode(QToolButton.InstantPopup)

        self.actionToolButton = self.iface.addToolBarWidget(self.toolButton)

        self.taskManager = QgsApplication.taskManager()

        self.signalling.log(0,'QGisPluginGas.__init__()','Plugin Loaded')
            
    def unload(self):
        self.iface.removePluginVectorMenu(vectorMenuText, self.action_GasFileScratch)  # submenu name, menuItem
        self.iface.removePluginVectorMenu(vectorMenuText, self.action_GasFileGeopkg)  
        self.iface.removePluginVectorMenu(vectorMenuText, self.action_LiveLink)  

        self.iface.removeToolBarIcon(self.actionToolButton )
        
        if self.glmClient:
            self.glmClient = None


    # start GLM live link
    def connectLiveLink(self): 
        #self.signalling.log(3,'Plugin.connectLiveLink()', f'CALLED. trace :\n{"".join(traceback.format_stack())}')
        if (not self.glmClient.connected) :  # first time, or we have a disconnected client
            self.glmClient.startListener() # start listening for messages
                       
        else:  # already running - can this still happen ?
            self.signalling.log(3,'Plugin.connectLiveLink()', f'Live link plugin Already Running')
 
    def disconnectLiveLink(self):      
        self.glmClient.stopListener() 
          
 
    # called when we have successfull connection or disconnection : adjust UI to reflect new state of self.glmClient.connected
    # we toggle which handler will respond to the UI action, and the UI text
    def on_ToggleLivelinkConnect(self):
        self.signalling.log(3,'Plugin.on_ToggleLivelinkConnect()', f'Connected :{self.glmClient.connected}')
        if (not self.glmClient.connected):  # we are NOT connected so UI action should switch to "Connect"
            self.action_LiveLink.setText("Live Link") 
            self.reconnectSignal(self.action_LiveLink.triggered,self.connectLiveLink) # connect runLive()
        else: # we ARE connected so UI action should switch to "DisConnect"
            self.action_LiveLink.setText("Disconnect Live Link")  
            self.reconnectSignal(self.action_LiveLink.triggered,self.disconnectLiveLink) # connect disconnectLiveLink()


# wire up a signal handler, disconnecting any previous handler
    def reconnectSignal(self,signal, newhandler=None):
        signal.disconnect() # discconnect any prev
        if newhandler is not None: # add new one
            signal.connect(newhandler)
 
 
# Gasfile loader entry points
    def runLinear(self):
        self.legendType = 'LINEAR'
        self.layerType = 'GEOPACKAGE'
        self.runFiles()
        return

    def runScratch(self):
        self.legendType = 'LINEAR'  # todo resolve user choice here
        self.layerType = 'SCRATCH'
        self.runFiles()
        return

# load a gas file(s) supports multiple files selected from dialogue
    def runFiles(self):
        
        #Get file selection from user
        path = str(QgsSettings().value('ioGAS-Qgis/lastInputDirectory', Path.home()))
        dlg = QFileDialog()
        dlg.setFileMode(QFileDialog.ExistingFiles)
        dlgResponse=dlg.getOpenFileNames(self.iface.mainWindow(), "Open ioGAS files", path, "GAS (*.gas)")
        selFileNames= dlgResponse[0] # nb latest version of dialogue returns a 2-tuple of (filenamelist , filter)

        # remember last used location
        if len(selFileNames)>0:
            QgsSettings().setValue('ioGAS-Qgis/lastInputDirectory', os.path.dirname(selFileNames[0]))

        # load each selected file
        for filePath in selFileNames:
            self.runFile(filePath)

# load a single gas file
    def runFile(self, filePath):
        self.signalling.log(3,'Plugin.runFile()', f'Loading filePath:{filePath}')
        
        fileLayerName = os.path.splitext(os.path.basename(filePath))[0]  # filename only w/out extension
        
        # Read the Gas file
        gasdata = GasData_File(filePath,self.signalling)        
        try :
            if not gasdata.load_data():
                self.signalling.notify(NotifyLevel.Warning,f'Unable to display ioGAS File : {gasdata.validationMessage}')
                return

            if gasdata.validationMessage :  # we loaded but there were warnings
                self.signalling.notify(NotifyLevel.Warning,f'File Loaded, with warning : {gasdata.validationMessage}')

        except ValueError as err:    # a handled error reading gas file data, can't display layer
            self.signalling.notify(NotifyLevel.Warning,f'Unable to display ioGAS File : {filePath}\n{err.args[0]}')
            return
        except Exception as err:
            self.signalling.notify(NotifyLevel.Warning,f'Unable to read from ioGAS File : {filePath}\n{err.args}')
            return

        # Add gas data as a vector qgis layer

        gasLayerLogic = GasLayerLogic_File(self.signalling)
        gasLayerLogic.setLegendType(self.legendType)

        #check for legend limitation (Qgis Ver vs Max number of unique combinations for linera legends)
        if not gasLayerLogic.canDisplayLegend():
            msg = f'Cannot display this file - must be fewer than {app.LEGEND_LIMIT} unique attributes for Linear Legend'
            self.signalling.notify(NotifyLevel.Critical,msg)
            return

        # Option 1: Create a Geopackage layer at same location as source gas file , 
        if self.layerType == 'GEOPACKAGE' :
            dest = os.path.normpath(os.path.join(os.path.dirname(filePath), os.path.splitext(filePath)[0] + ".gpkg")) # geopackage filepath from dir + filename w/out ext + .gpkg
            layer = gasLayerLogic.createGeoPackageLayer(dest,fileLayerName,gasdata)

        # Option 2 : Create as a Memory-Only Layer
        if self.layerType == 'SCRATCH':
            fileLayerName = fileLayerName + " [Scratch]"
            layer = gasLayerLogic.createMemoryLayer(fileLayerName,gasdata)


        # add layer to the map
        if layer.isValid():
            QgsProject.instance().addMapLayer(layer)
            if len(gasdata.warning) > 0:
                self.signalling.notify(NotifyLevel.Warning,f'This operation raised the following warning : {gasdata.warning} {filePath}')
            else:
                self.signalling.log(3,'Plugin.runFile()', f'Loaded OK:{filePath}')
        else:
            self.signalling.notify(NotifyLevel.Warning,f'Failed to load file : {filePath}')



# Logging & Notification 
# UI-utilising QT signals as we are sometimes crossing thread boundaries.
# These event handlers are wired up to subscribe to the QT signals declared in our Signalling_App class

    #@pyqtSlot(object,object)
    def on_notify_raised(self, threadName, notifylevel,msg):
           
        msg = msg.replace("<","").replace(">","").replace("\"", "")  #in case we are reporting xml -  replace <> to avoid confusing QgsMessageLog!
        self.on_log_raised(threadName,0,'Notify',msg)
        if  notifylevel == NotifyLevel.Success:
            self.iface.messageBar().pushMessage('Success', msg, level=Qgis.Success, duration=3)
        elif  notifylevel == NotifyLevel.Info:
            self.iface.messageBar().pushMessage('Info', msg, level=Qgis.Info, duration=3)
        elif  notifylevel == NotifyLevel.Warning:
            self.iface.messageBar().pushMessage('Warning', msg, level=Qgis.Warning, duration=3)
        elif  notifylevel == NotifyLevel.Critical:
            QMessageBox.warning(None, 'Error', msg) # a dialogue instead for critical stuff
        else:
            return
        return

 #@pyqtSlot(object,object,object)
    def on_log_raised(self, threadName, loglevel,source,msg):

        if (loglevel not in LOG_LEVELS):
            return   
            
            
        # write critical messages to file
        if loglevel>=0 : # TODO currently log all messages to file regardless of LOG_LEVELS
            try:
                timestampStr = datetime.now().strftime("%d-%b %H:%M:%S")
                fileL = open(self.logFilePath,"a") 
                fileL.write(f'[{threadName}|{timestampStr}][{source}]:{msg}\n')
                fileL.close()
            except:
                pass # swallow if unable to log to file
                
 
        # As this is the UI version, we also write all messages to qgis UI (message log panel)
        msg = msg.replace("<","").replace(">","").replace("\"", "")  #in case we are reporting xml -  replace <> to avoid confusing QgsMessageLog!
        QgsMessageLog.logMessage(f'[{threadName}] {source} {msg}', APP_TAG, level=Qgis.Info) #TODO : change qgis log level with loglevel?.
        

