# Misc support functionality 
# No dependancies

import sys
import os
from pathlib import Path

import configparser
from enum import Enum, auto
import time
import threading 

# Some utility functions

LOG_FILE = 'plugin.log.txt'

# Will only display levels in this this list; 
# suggested convention : 0=prod,user info; 1=Prod, user warning; 2=Dev-warning/info; 3=Dev-specific debug nsg
LOG_LEVELS = [0,1,2,3] # = [0] or  [0,1] for production, = [3] to just get a specific debug message

CONFIG_FILE = 'settings.ini'
CONFIG_PATH = 'ioGAS.QGIS'
PLATFORM_WIN = 'WIN'
PLATFORM_MAC = 'MAC'

DEFAULT_PORT = '2718'

APP_TAG = 'ioGAS'
LIVELINK_LAYERNAME = " [ioGAS live]"
LIVELINK_ROWIDFIELDNAME = "ioGAS_RowID__"


# resolve path to logfile, create folder if not existant, hill_prev : whether to kill any previous file at that path
def get_logfile_path(kill_prev=False):
    logFilePath = get_config_folder() # same as config
    logFilePath =  os.path.join(logFilePath, LOG_FILE) # add filename to path
    
    if kill_prev :    # Kill any previous logfile
        try:
            os.remove(logFilePath)
        except OSError:
            pass

    return logFilePath




# read/create Config file
def read_or_initialise_config():
    config = configparser.ConfigParser()
    
    # create folder if does not texist
    configpath = get_config_folder()
        
    # if config file does not exist initialise one with default values
    configpath = os.path.join(configpath, CONFIG_FILE)  # add filename to path
    if not os.path.exists(configpath): # create default config file.
        config['DEFAULT'] = {}
        config['LIVELINK'] = {'port': DEFAULT_PORT}
        with open(configpath, 'w') as configfile:
            config.write(configfile)
            msg=f"Initialising config file at : {configpath}"
    else:
        # read existing one
        config.read(configpath)        
        msg=f"Config file found at : {configpath}"  
        
    return config, msg

def get_config_folder():
    # create folder if does not exist 
    configfolder = ''
    try:
        if get_os_platform()==PLATFORM_WIN:
            configfolder = os.path.join(os.getenv('APPDATA'), CONFIG_PATH)  # windows user appdata,roaming
        else:
            # APPDATA no good on mac, so use user-specific ~/Library/Application Support
            configfolder = os.path.join(os.getenv('HOME'),'Library','Application Support', CONFIG_PATH)  # app prefs off mac user home
    except:
        raise Exception("ERROR Cannot locate preferences folder")

    #print(f'configfolder:  {configfolder}')
    if not os.path.exists(configfolder):
        os.makedirs(configfolder)
    return configfolder

# get os platform as either MAC or WINDOWS
def get_os_platform():
    if os.name == 'posix' :
        return PLATFORM_MAC
    else :
        return PLATFORM_WIN

        
#  lookup data value by index (todo slow?)
def lookup(line, cols):
    ret = []
    for col in cols:
        index = col.index  
        value = line[index]
        fixed  = fixNonNumeric(col, value)
        ret.append(fixed)
    return tuple(ret)
    
# if field type should be numeric then we try to parse to float or int, or leave null    
def fixNonNumeric(column, value):
    if (column.type=='Numeric'):
        try:
            if '.' not in value: # int
                value = int(value)
            else: # float
                value = float(value)
        except ValueError: # proably text
            value = None
    # leave text type field values unchanged 
    return value            
     
# return a 24-bit [gas,java-compatible] ARGB int value as tuple of r,g,b,a
def RGBAfromInt(argb_int):
    blue =  argb_int & 255 # take least sig byte
    green = (argb_int >> 8) & 255 # take next byte by shifting right
    red =   (argb_int >> 16) & 255
    alpha = (argb_int >> 24) & 255
    return (red, green, blue, alpha)

 
# gas escaping rules
def unescape(cell):
    ret =  cell.replace("_a", "&")
    ret = ret.replace("_l", "<")
    ret = ret.replace("_g", ">")
    ret = ret.replace("_q", "\"")
    ret = ret.replace("_c", ",")
    ret = ret.replace("_t", "\t")
    ret = ret.replace("_u", "_")
    return ret

# replace illegal chars with substitutions
def escape(cell):
    ret =  cell.replace("&","_a")
    ret = ret.replace("<","_l")
    ret = ret.replace(">","_g")
    ret = ret.replace("\"","_q")
    ret = ret.replace(",","_c")
    ret = ret.replace("\t","_t")
    ret = ret.replace("_","_u")
    return ret
    
# already have a column of this name?
def containsName(cols, colName):
    for c in cols:
        if c.name == colName:
            return True
    return False

# convert between GLM object and XML 
def convert(oMsg):
    # convert g to a string (via file - yuck)  : Serialise to XML ?

    p = get_config_folder()
    f = os.path.join(p,  "glm.message.temp.txt")
    #self.signalling.log(1,'', f'writing to {f}')
    file = open(f, "w")
    oMsg.export(file, 0)
    file.close()
    file2 = open(f, "r")
    strMsg = file2.read()
    file2.close()
    return strMsg

       
class NotifyLevel(Enum):  #  UI message severity enum
    Success = auto()
    Info = auto()
    Warning = auto()
    Critical = auto()



class TimerError(Exception):
    """A custom exception used to report errors in use of Timer class"""

# simple timer class for tracking performance
class Timer:
    
    def __init__(self,pSignalling):
        
        # we wire up signals  that should trigger our own handlers
        self.signalling = pSignalling  
        self.signalling.sig_timer_start.connect(self.start)
        self.signalling.sig_timer_stamp.connect(self.stamp)
        self.signalling.sig_timer_printlog.connect(self.printlog)
        self.signalling.sig_timer_stop.connect(self.stop)
        
        self.log = '' # keep track of timing messages, view on stop
        self.start_time = None
        self.prev_time = None

    def start(self,msg=''):
        if self.start_time is not None:
            self.stamp(msg + '(Timer Starting)') # just stamp if already running
            #raise TimerError(f"Timer is running. Use .stop() to stop it") # WILL THIS BE CAUGHT ? should write to log
        self.log = ''
        self.start_time = time.perf_counter()
        self.prev_time= self.start_time
        self.stamp(msg + '(Timer Starting)')

    def stamp(self,msg=''):
        if self.start_time is None:
            self.start(msg) # auto start if req'd on first stamp
            #raise TimerError(f"Timer is not running. Use .start() to start it")
            
        elapsed_time = time.perf_counter() - self.start_time
        prev_elapsed_time = time.perf_counter() - self.prev_time          
        self.log = f'{self.log }\n[{elapsed_time:0.4f}][{prev_elapsed_time:0.4f}]:{msg}'
        self.prev_time = time.perf_counter()
    
    def printlog(self,msg=''):
        self.signalling.log(3,'TIMER',f'{self.log}')
     
    def stop(self,msg=''):

        self.stamp(msg + ' (Timer Stopping)')
        self.start_time = None
        self.prev_time = None
        self.signalling.log(3,'TIMER',f'{self.log}')

        
     


