import sys
import os
from pathlib import Path
import tempfile

import threading 
import socket
from io import StringIO

from time import sleep
import select

try:
    from ioGAS.gassupport import *
except:
    from gassupport import *


CHARNULL = '\0'
BUFF_SIZE = 4096


class GasLinkComms():

    def __init__(self, pSignalling, pConfig, pHangTesting):
        self.config = pConfig
        self.signalling = pSignalling 
      
        self.port=int(self.config['LIVELINK']['port'])
       
        self.sock = None
        self.userRequestedCancel = False
        
        # for a unit test
        self.hangTesting = pHangTesting
        if self.hangTesting:
            self.signalling.log(1,'GasLinkComms.connect()', 'WARNING - HANG TESTING CLIENT - WILL FREEZE IF # RECEIVED')


    def connect(self):
        # Initialise tcp/ip
        self.signalling.log(1,f'[GasLinkComms.connect()] port={self.port}', '')
        server_address = ('localhost', self.port)
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            sock.connect(server_address)           
            self.signalling.log(1,'GasLinkComms.connect()', 'tcp/ip socket initialised')
        except:
            self.signalling.notify(NotifyLevel.Critical, f'Could not start Live Link on {str(server_address)}\n(you may need to start ioGAS)') 
            sock.close()
            raise (ValueError('Failed to connect socket'))
        return sock
    
    # send a GLM Message object
    def sendMessage(self, strMsg):
        self.signalling.log(1,'GasLinkComms.sendMessage()', 'init')
        txBuff = StringIO()
        txBuff.write(strMsg)
        txBuff.write(CHARNULL)
        bytes = txBuff.getvalue().encode()
        self.sock.sendall(bytes)
        self.signalling.log(1,'GasLinkComms.sendMessage()', 'msg Sent')
        # close txBuff ?

    def stopListener(self):
        try:
            self.userRequestedCancel = True # signal to listener thread that we want to stop listening
            self.signalling.log(2,'GasLinkComms.stopListener()', 'Disconnection Requested')
            
        except:
            self.signalling.log(2,'GasLinkComms.stopListener()', 'FAILED')  
        
    # this function allows the inner listener loop to check the status of the 
    # self.userRequestedCancel flag allowing external interruption of the listener   
    def checkForUserCancel(self):
        return self.userRequestedCancel
        
    # Connect! and start listener! 
    def startListener(self):
        self.sem = threading.Semaphore(0) # for tests
        try:
            self.sock = self.connect()
        except:
            raise (ValueError('Failed to connect socket'))
            
        try:
            self.userRequestedCancel = False # reset 
            self._thread = threading.Thread(target=self.listen, args =(self.checkForUserCancel, ))   # kick off the listen() method on a new thread, pass in function reference to check for user cancel flag
            self._thread.daemon = True  # daemonic threads allow the main program to exit even if this thread is still running. main prog exit will kill this thread.      
            self._thread.start()
        except:
            raise (ValueError('Failed to connect socket'))

    
    # for use by tests that want to block until messages are in
    def waitForMessages(self, n):
        for i in range(n):
           # self.signalling.log(1,'waitForMessages', f'into  ACQ {i}')
           self.sem.acquire(timeout = 10) #  (10s timeout)
           # self.signalling.log(1,'waitForMessages', f'out of ACQ {i}')
           
        
    # main tcp/ip listening loop - gets run on a background thread
    def listen(self,pCheckForUserCancel):  # pCheckForUserCancel is a function ref that will check for outside request to cancel (disconnect)
        try:
            self.signalling.log(2,'GasLinkComms.listen()', '')  
            msgStarted= False # flag when we start receiving bytes for a new message 
            blkCount=0
            streamEnd  = False
            cancelledByUser = False
            rxBuff = StringIO()
            
            # loop checking for stream end or user cancel
            while not (streamEnd or cancelledByUser):
                #block on select call waiting for next byte, OR timeout
                readyread,x1,x2 = select.select( [self.sock], [], [], 1 ) #Once sec timeout
                
                if not readyread:
                    # no sockets in list, so no data to process, ie expected timeout so loop (after checking for user cancellation)
                    #self.signalling.log(2, 'GasLinkComms.listen()', 'TIMEOUT')
                    if (pCheckForUserCancel()): 
                        self.signalling.log(2, 'GasLinkComms.listen()', 'User Cancel Request detected')
                        cancelledByUser = True
                else:
                    if not msgStarted : # we are starting new message - start/stamp timer
                        ##TIMER
                        #self.signalling.timer_stamp('New Message started')
                        msgStarted = True
                        blkCount=0
                    
                    # there is data so read data into a python bytes object
                    bytes = self.sock.recv(BUFF_SIZE) # returns zero bytes iff connection closed
                    # buffsize vs time profiling: 1 30s, 100 5s,  10000 5s (17k row geochem)
                    blkCount += 1

                    if (len(bytes)==0):
                        streamEnd = True #gas disconnected
                        msgStarted = False
                        self.signalling.log(2,'GasLinkComms.listen', "STREAM ENDED")  
                    else:
                        # read byte at a time checking for end of message
                        for byteint in bytes:
                        
                            if ((byteint > 31) and (byteint < 128)) or (byteint == 0):
                                # we allow printable ascii or null only
                                character = chr(byteint)
                            else:
                                self.signalling.log(1, 'GasLinkComms.listen()', f'illegal byte {byteint}  to  _')
                                character = "_"  # nonprintable ascii OR utf problem workaround
                                
                            # THIS ONLY IS FOR A UNIT TEST THAT NEEDS CLIENT TO SIMULATE CRASHING
                            if (self.hangTesting and character == "#"):
                                self.signalling.log(1, 'GasLinkComms.listen()', f'!!! deliberate hang after getting #')
                                while True: # CRASH...
                                    pass
                                
                            if (character != CHARNULL): 
                                rxBuff.write(character)                            
                            
                            else:   
                                # we got a terminated message, grab message and raise signal 
                                # but do not append  - some xml parsers get upset?
                                strMsg = rxBuff.getvalue()
                                
                                # Logging, timing :
                                self.signalling.log(3,'GasLinkComms.listen()', f'got message :\n{strMsg[0:100]}\nBlock Count:{blkCount}\nRaised On thread: {threading.current_thread().name}')
                                #self.signalling.timer_stamp(f'GLMClient.listen(): Got message :\n{strMsg[0:100]}\nBlock Count:{blkCount}')  ##TIMER
                                
                                self.signalling.raise_message_received(strMsg)
                                msgStarted = False # reset for next msg
                                rxBuff = StringIO() # must clear, so we can keep looping and adding
                                
                                # release semaphore used for unblocking in tests
                                if (self.sem): #ie if not None
                                    # self.signalling.log(3,'GLMClient.listen()', f'about to release test sem')
                                    self.sem.release()

            # connection has ended, tidy up
            self.sock.close()
            rxBuff.close()

            if cancelledByUser:
                self.signalling.log(3,'GasLinkComms.listen', f'user_cancelled streamEnd {streamEnd}')
                self.signalling.raise_user_cancelled() # raise signal        
            else: # should we check stream end true instead here?
                self.signalling.log(3,'GasLinkComms.listen', f'connection_lost streamEnd {streamEnd}')
                self.signalling.raise_connection_lost() # raise signal
            
        except Exception as e:
            self.signalling.raise_listener_exception(e) # raise signal   
            raise  

    
        

