import sys
import os
from pathlib import Path
import tempfile
import traceback

import threading 
import socket
from io import StringIO

from time import sleep
import configparser


try:
    import ioGAS.gas as gas #used by gaslink
    import ioGAS.gaslink as gaslink
    from ioGAS.gaslinkcomms import GasLinkComms
    from ioGAS.gasdata_link import GasData_Link
    from ioGAS.gassupport import *
    
except:
    import gas as gas #used by gaslink
    import gaslink as gaslink
    from gaslinkcomms import GasLinkComms
    from gasdata_link import GasData_Link
    from gassupport import *

REQUIRED_MINOR_VERSION = 5 # gaslink protocol version
buildID  = "202054" # build number
majorVersion = "8.2" # major version (aka branch, eg 7.3)

class GasLinkClient_Base():
    CLIENT_TYPE = "QGIS"
    
    def __init__(self,pSignalling, pConfig, pClientType, pHangTesting): #,pData = None, pComms = None):
        self.signalling = pSignalling 
        self.config = pConfig
        self.clientType = pClientType
        
        self.signalling.log(3,'GasLinkClient_Base.init()', f'init')
    
        self.gasData = None # format-independant rep of gas data and attributes
        self.comms = None  # manages tcp/ip and xml-level messging
        self.connected = False
        
        self.hangTesting = pHangTesting
        
        self.LinkFileName = "" # currentl data filename opened on livelink,  we retain this state between redraws  
       
    def setup(self):
        # setup data and tcp/ip comms classes 
        self.gasData = GasData_Link(self.signalling,self.clientType)   # a data object to hold received data
        self.comms = GasLinkComms(self.signalling, self.config, self.hangTesting) # a tcp/ip comms -level message management            
   

    def startListener(self):
        try:
            self.comms.startListener()  # Connect and start listening
        except Exception as err:
            self.signalling.log(3,'GLMClient.startListener()', f'Unable to start listener : {err.args}')
            self.connected = False
            return
            
        self.sendGLMConnect() # start the message-level protocol
        
    # for unit tests, that will wait for n messages, and then return    
    def waitForMessages(self, n):
        self.signalling.log(3,'GLMClient.waitForMessages()', f'waiting for {n} messages')
        self.comms.waitForMessages(n)
        self.signalling.log(3,'GLMClient.waitForMessages()', f' - all received')

    def stopListener(self):
        try:
            self.comms.stopListener()  # flag comms to stop listening; this will trigger a connection lost
            #self.connected = False # we will be alerted when the listener loop stops

        except Exception as err:
            self.signalling.log(3,'GLMClient.stopListener()', f'Unable to s listener : {err.args}')
            return
            
########  SEND METHODS  ###########

   # send a GLM Message object
    def sendMessage(self, oMsg):        
        self.signalling.log(1,'GLMClient.sendMessage()', 'init')
        strMsg = self.convert(oMsg)  # convert GLM message object to string
        self.comms.sendMessage(strMsg)

    # send a GLMConnect message
    def sendGLMConnect(self):
        g = gaslink.GLMConnect() 
        protocol = gaslink.Protocol()
        g.set_protocolVersion(protocol.Version) # 
        g.set_protocolMinorVersion(protocol.MinorVersion) # 
        g.set_clientType(self.clientType) 
        
        self.sendMessage(g)
        self.signalling.log(1,'GLMClient.sendGLMConnect()',f'GLMConnect Sent, protocol ver:{protocol.Version}/{protocol.MinorVersion}')
        
    
    # send list of views as a  GLMGetViewListDone
    def sendGLMGetViewListDone(self):
        g = gaslink.GLMGetViewListDone()
        views = self.getViewList()
        for v in views:
            g.add_view(v)
        
        self.sendMessage(g)
        self.signalling.log(1,'GLMClient.sendGLMGetViewListDone()','sendGLMGetViewListDone Sent')
        
    # send a view (ie a table of data)    
    def sendGLMGetDataDone(self, viewName):
        dataMessage = self.getDataMessage(viewName)
        #self.signalling.log(1,'GLMClient.sendGLMGetDataDone()', f'DEBUG:dataMessage:\n{self.convert(dataMessage) }')
        self.sendMessage(dataMessage)
        self.signalling.log(1,'GLMClient.sendGLMGetDataDone()', 'sent')

    # send a GLMApplyAttributes message to gas : list of selected feature ROW ID's
    def sendGLMApplyAttributes(self, selFeatureRowIDs) :
        # selFeatureRowIDs is list of int rowid values as strings - eg ['177', '176', '175']
        #self.signalling.log(3,'GLMClient.sendGLMApplyAttributes()', f'selFeatureRowIDs:{selFeatureRowIDs}')
        try:
            if selFeatureRowIDs ==  None :
                self.signalling.log(3,'GLMClient.sendGLMApplyAttributes()', f'Null selFeatures')
                return
            
            # convert list of int ROW ID's to csv string of ints eg '177,176,175'
            csvFeatures = ",".join( map( str, selFeatureRowIDs ) ) 
            
            g = gaslink.GLMApplyAttributes()
            g.set_useGasState(True)
            g.set_rowIDs(csvFeatures)
            
            self.sendMessage(g)
            self.signalling.log(1,'GLMClient.sendGLMApplyAttributes()',f'Sent')
        
        except Exception as err:
            self.signalling.log(3,'GLMClient.sendGLMApplyAttributes()', f'Exception : {err.args}')
        
 
    def on_listener_exception(self,e):
        self.signalling.log(1,'GLMClient_Base.on_listener_exception()', f'LISTENER EXCEPTION :\n{e.args}')
        self.connected = False
        self.invalidate_data() 
        
    def on_connection_lost(self):
        self.signalling.log(1,'GLMClient_Base.on_connection_lost()', f'CONNECTION LOST')
        self.connected = False
        self.invalidate_data()   

    def on_user_cancelled(self):
        self.signalling.log(1,'GLMClient_Base.on_user_cancelled()', f'USER CANCELLED')
        self.connected = False
        self.invalidate_data()

    # invalidate data so that GIS layer will be removed
    def invalidate_data(self):
        self.gasData.initialise()
        self.gasData.set_data_message(None)
        self.gasData.set_attr_message(None) 
        self.load_data()  

########  RECEIVE METHODS  ###########  

    #@pyqtSlot(object)
    def on_selection_changed(self,selFeatures):
        self.signalling.log(1,'GLMClient_Base.on_selection_changed()', '')
        try:
            self.sendGLMApplyAttributes(selFeatures)
        except ValueError as e:
            # RW is this the place to do this? what about other errors?
            self.signalling.notify(NotifyLevel.Critical, str(e)) 

    #@pyqtSlot(object)            
    def on_message_received(self, strMsg):

        try:
            oMsg = self.parse(strMsg) # deserialise xml to obj
            self.signalling.log(1,'GLMClient_Base.on_message_received()  ', f'Received msg of type : {type(oMsg)}')
            # route to appropriate message handler based on message type 
            # nb - uses a dictionary of callables keyed off msg class type (a py switch alternative)
            options = {gaslink.GLMConnectDone: self.handle_GLMConnectDone, \
                gaslink.GLMSendData: self.handle_GLMSendData, \
                gaslink.GLMSendAttributes: self.handle_GLMSendAttributes, \
                gaslink.GLMGetViewList: self.handle_GLMGetViewList, \
                gaslink.GLMGetData: self.handle_GLMGetData }

            ff = options.get(type(oMsg))
            if (ff):
                ff(oMsg)
                self.signalling.log(1,'GLMClient_Base.on_message_received()  ', f'Processed : {type(oMsg)}')
            else:
                self.signalling.log(1,'GLMClient_Base.on_message_received()  ', f'Ignored : {type(oMsg)}')
        except Exception as ex:
            self.signalling.log(1,'GLMClient_Base.on_message_received()  ', f'Error processing {str(ex)}')

            
    def handle_GLMConnectDone(self,oMsg):
        self.signalling.log(1,'GLMClient.handle_GLMConnectDone()', '')
        connectDone = oMsg
        ok = connectDone.get_connectionOK()
        maj = connectDone.get_ServerVersion()
        min = int(connectDone.get_ServerMinorVersion())
        desc = connectDone.get_description()
        if ok:
            # we do not need to check major version as gas will have
            # but gas will allow wrong minor version, so we have to check
            # that gas is recent enough (epsg was added in 0.2.5)
            if (min < REQUIRED_MINOR_VERSION):
                # WARNING ONLY (this code path can't run in 7.3 release but its for future)
                self.signalling.log(1, f'Version error, ioGAS too old {maj}  {min}', '')
                # Tell them gas version needs to be our major version or later. 
                # (major version is in sync with gas when built)
                # NB the comparison is on 'protocol version' as we can't know the version of gas we are connected to
                self.signalling.notify(NotifyLevel.Warning, f'Please update your version of  ioGAS to {majorVersion} or later.')
                # NOTE If we decide that the gas minor version is too old
                # we could disconnect like below, but for now we work best we can
                
            self.connected = True
            self.signalling.raise_connected() # raise the connected signal 
            self.signalling.log(2, 'handle_GLMConnectDone',f'connect done OK, ioGAS is GLM {maj}  {min}')
        else:
            # IMPORTANT we rely on this rejection from gas.  
            # gas also closes the socket, but detecting that alone seems not to work
            self.signalling.log(2, 'handle_GLMConnectDone',f'rejected by ioGAS {desc}')
            self.signalling.notify(NotifyLevel.Critical, f'Connection rejected. Please update your version of  ioGAS to {majorVersion} or later.')

            # we don't set self.connected = False # will be reset when listener stops
            self.stopListener()

 
    def handle_GLMGetViewList(self, oMsg):
        self.signalling.log(1,'GLMClient.handle_GLMGetViewList()', '')
        self.sendGLMGetViewListDone()
        
    def handle_GLMGetData(self, oMsg):
        viewName = oMsg.get_viewName()
        self.signalling.log(1,'GLMClient.handle_GLMGetData()', f'request for {viewName}')
        self.sendGLMGetDataDone(viewName)

    # Gas has sent data         
    def handle_GLMSendData(self,oMsg):
        self.signalling.log(1,'GLMClient.handle_GLMSendData()', '')

        self.signalling.log(3,'GLMClient.handle_GLMSendData()', f'stateNull={oMsg.get_stateNull()}  name ={oMsg.get_name()}')
        oMsg.get_stateNull()  # TODO ??
        if oMsg.get_stateNull():  # statenull true means  no data current
            #self.log(3,'GLMClient.handle_GLMSendData()', 'NULL SENDDATA MESSAGE RECEIVED')
            self.gasData.initialise()
            self.gasData.nullMessage = True
            self.gasData.set_data_message(None)
            self.gasData.set_attr_message(None)  #we should probs clear previous attributes so not wrongly applied to any new data
        else:
            #self.signalling.log(1,'GLMClient.handle_GLMSendData()', f'msg:\n{self.convert(oMsg) }')# convert GLM message object to string
            self.gasData.nullMessage = False
            self.gasData.set_data_message(oMsg)
        
        self.load_data()     # try to load data   

    # Gas has sent/updated attributes       
    def handle_GLMSendAttributes(self,oMsg):
        self.signalling.log(1,'GLMClient.handle_GLMSendAttributes()', '')
        #self.signalling.log(1,'GLMClient.handle_GLMSendAttributes()', f'msg:\n{self.convert(oMsg) }')# convert GLM message object to string

        self.gasData.set_attr_message(oMsg)
        self.load_data()     # try to load data   

    # Load data and attributes into a GasData object then display  
    def load_data(self):

        # It is possible to recieve messages that are in an inconsistent
        # state with respect to each other
        try:
            ##TIMER
            #self.signalling.timer_stamp('gasData.load_data.start')
            
            self.gasData.initialise() # clear any previously received data
            datavalid = self.gasData.load_data()
            
            ##TIMER
            #self.signalling.timer_stamp('gasData.load_data.done')
            
            self.display_data(datavalid)
            
        except  Exception as err:
            self.signalling.log(3,'load_data()', f'unable to display\r\n{traceback.format_exc()}')


#### Support Methods ####

    # convert between GLM object and XML 
    def convert(self,oMsg):
        # convert g to a string (via file - yuck)  : Serialise to XML ?

        p = get_config_folder()
        f = os.path.join(p,  "glm.message.temp.txt")
        self.signalling.log(1,'', f'writing to {f}')
        file = open(f, "w")
        oMsg.export(file, 0)
        file.close()
        file2 = open(f, "r")
        strMsg = file2.read()
        file2.close()
        return strMsg


    def parse(self,strMsg):
        # remove xml header, then parse
        strMsg = strMsg.replace('<?xml version="1.0" encoding="UTF-8" standalone="yes"?>', '')
        oMsg = gaslink.parseString(strMsg, True)
        return oMsg






