import sys
import os
import traceback
from pathlib import Path
import tempfile

import threading 
import socket
from io import StringIO

from time import sleep
import configparser

from qgis.core import * 
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMenu,QToolButton,QMessageBox


from ioGAS.gaslinkclient_base import GasLinkClient_Base
from ioGAS.gasdata_link import GasData_Link
from ioGAS.gaslayerlogic_link import GasLayerLogic_Link
from ioGAS.gaslinkcomms import GasLinkComms
from ioGAS.gassupport import * #for constants

#for building messages
import ioGAS.gas as gas
import ioGAS.gaslink as gaslink


    
# a GLM Client class that orchestrates between the Comms and Data objects to manage a glm protocol dialogue, and manages the live layer in gis environment.
class GasLinkClient_App(GasLinkClient_Base): 

    def __init__(self,pSignalling,pConfig,iface):
        super().__init__(pSignalling,pConfig, GasLinkClient_Base.CLIENT_TYPE, False) 
        self.iface = iface

        # We use QT signals to allow marshalling of notfications and events between the listener which is on a seperate thread, and other components 
        # so here we wire up our signallings signals to our handlers (aka slots)

        self.signalling.sig_message_received.connect(self.on_message_received)     #  message handler 
        self.signalling.sig_selection_changed.connect(self.on_selection_changed)  
        
        self.signalling.sig_connection_lost.connect(self.on_connection_lost)
        self.signalling.sig_user_cancelled.connect(self.on_user_cancelled)
        self.signalling.sig_listener_exception.connect(self.on_listener_exception)        

        self.signalling.log(3,'GasLinkClient_App.__init__()', f'')

        self.setup()
        
        # a qgis/gaslayer management object
        self.gasLayerLogic = GasLayerLogic_Link(self.signalling)
        self.gasLayerLogic.setLegendType('LINEAR')
        
# Display the new/refreshed data    (RW this is only override)
    def display_data(self,datavalid):
        self.signalling.log(2,'GLMClient.display_data()',f'')

        # if we do not have valid data then we should remove this layer
        # reminder this can get called when we have not yet received attributes
        if not datavalid:
            self.signalling.log(2,'GLMClient.display_data()',f'Data not Valid removing any live layer')
            self.gasLayerLogic.removeLiveLayerIfExists() # remove ANY previous live layer
            self.iface.mapCanvas().refreshAllLayers() # force map to re-display
            return


        # We will try to retain previous layer labelling if we appear to be redrawing the same data files as was previously open on the live link
        # Prior to drawing the new layer, we grab any existing layer's labelling settings
        # we attempt to re-apply labelling if we seem to be re-displaying the same dataset
        # self.LinkFileName holds previous state of live link dataset filename - we maintain state of this between redraws 
        oldLayer = self.gasLayerLogic.findLiveLayer()  # grab existing(prev) live layer , ignored if no prev
        palSettings = QgsPalLayerSettings()
        isRuleBased = False
        oldLabelingEnabled = False

        rootRuleCopy = None

        retainLabelState = (self.LinkFileName != "") and (self.LinkFileName == self.gasData.name) and oldLayer != None
        if retainLabelState :  
            oldLabelingEnabled = oldLayer.labelsEnabled()
            self.signalling.log(2,'GLMClient.display_data()',f'Labelling : Getting previous state from : {self.LinkFileName},oldLabelingEnabled:{oldLabelingEnabled}')
            oldLabelling = oldLayer.labeling()

            if oldLabelling != None : # we have previous labelling - either simple or rules based 
                # we try to grabe labelling from old layer's labelling state
                self.signalling.log(2,'GLMClient.display_data()',f'oldLayer.labeling().type() : {oldLayer.labeling().type() }')
                
                palSettings = oldLayer.labeling().settings()
                isRuleBased = self.gasLayerLogic.hasRuleBasedLabeling(oldLayer)
                if isRuleBased:
                    rootRuleCopy = self.gasLayerLogic.copyRuleBasedLabeling(oldLayer)  # get a copy of rules from old layer

                self.signalling.log(2,'GLMClient.display_data()',f'Labelling rulebased?: {isRuleBased}')

        self.gasLayerLogic.removeLiveLayerIfExists() # remove ANY previous live layer
        self.iface.mapCanvas().refreshAllLayers() # force map to re-display
        
        # create new layer
        layerName = self.gasData.name + LIVELINK_LAYERNAME  
        newLayer = self.gasLayerLogic.createLiveLayer(layerName,self.gasData)  
        self.LinkFileName = self.gasData.name # remember state
        self.signalling.log(2,'GLMClient.display_data()',f'Created Live Layer {layerName}')

        # Re-apply previous labelling state to new layer if appropriate
        if retainLabelState :
            if oldLabelling == None or not oldLabelingEnabled  : 
                self.signalling.log(2,'GLMClient.display_data()',f'Labelling : Labelling was not enabled')  
                newLayer.setLabelsEnabled(False)  # looks like we need to set this explicitly else seems to re-nable previous state
            else :  # we have either simple or rules based labelling
                self.signalling.log(2,'GLMClient.display_data()',f'Attemptinmg to re-apply label styles for layer: {self.gasData.name}')
                self.signalling.log(2,'GLMClient.display_data()',f'settings type: {type(palSettings)}')
                if palSettings != None : 
                    newLayer.setLabeling(QgsVectorLayerSimpleLabeling(palSettings))
                    if isRuleBased:
                        self.gasLayerLogic.applyLabellingRules(newLayer,rootRuleCopy)
                    else : # simple
                        pass # nothing to do here?
                    newLayer.setLabelsEnabled(True)
                
                newLayer.triggerRepaint()


        # add layer to the map
        # TODO check whether we can in fact have multiple maps - in which case we might need to iterate.
        if newLayer.isValid():
            QgsProject.instance().addMapLayer(newLayer)
            self.signalling.log(2,'GLMClient.display_data()',f'LAYER VALID')
        else:
            self.signalling.log(1,'GLMClient.display_data()',f'Layer Create Failed')
            self.signalling.notify(NotifyLevel.Warning,'Layer Create Failed')
        
        ##TIMER
        #self.signalling.timer_stop('display_data.done')


# get list of appropriate layers currently open in qgis         
    def getViewList(self):

        # list of all the checked layers in the TOC (legend panel) # nb can't use map.getlayers() as does not reflect UI ordering
        root = QgsProject.instance().layerTreeRoot()
        layers = root.checkedLayers()
        #self.signalling.log(1,'GLMClient.getViewList()',f'layers:{len(layers)}')
    
        # keep only appropriate layers
        filteredLayers = list(filter(self.filter_IsVectorLayer,  layers))
        filteredNames = [layer.name() for layer in filteredLayers]
        
        #self.signalling.log(1,'GLMClient.getViewList()',f'filteredNames:{"[%s]" % ", ".join(list(map(str, filteredNames)))}')
        
        return filteredNames

    # a filter function / iterator - as each layer is passed to this function we return a true/false as to whether it passes the filter.
    def filter_IsVectorLayer(self,layer):

        self.signalling.log(1,'GLMClient.filter_IsVectorLayer()',f'layers:{type(layer)}: {layer.type()},{layer.name} ')
 
        # we only want point vector layers, ignore any existing live link layer
        if (layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PointGeometry 
            and not (layer.name().endswith(LIVELINK_LAYERNAME)) ):
            return True
        else:
            return False

            
    # build a GLMGetDataDone message from the named qgis layer
    def getDataMessage(self, layerName):

        self.signalling.log(2,'GLMClient.getDataMessage()',f'Layer Requested:{layerName}')
        
        try:
            # get specified layer
            vlayer = QgsProject.instance().mapLayersByName(layerName)[0]
            if vlayer is None:
                QMessageBox.warning(None, 'Error', f'Unable to find layer : {layerName}')  # report raised error details
                return

            # new glm message
            glmMsg = gaslink.GLMGetDataDone()
            glmMsg.set_stateNull(False)

            # Setup XSpecialColumns  
            spcolls = gas.xSpecialColumns()
            self.getSpecialColumns(vlayer, spcolls)
            #self.signalling.log(2,'GLMClient.getDataMessage()',f'spcolls:{str(spcolls.__dict__)}')
            glmMsg.set_specialColumns(spcolls)


            # setup columns
            cols = gas.xColumns()
            self.getXColumns(vlayer, cols)
            glmMsg.set_columns(cols)

            # fill row data from layer attribute data
            rows = gas.xDRs()
            self.getDataRows(rows, cols, vlayer)
            glmMsg.set_DRs(rows)

            self.signalling.log(1,'GLMClient.getDataMessage()', 'Message  Constructed')
  
        except  Exception as err:
            self.signalling.notify(NotifyLevel.Critical, f'GLMClient.getDataMessage ERROR : {err.args} ') 

        return glmMsg

    # convert attribute data to glm data rowss    
    def getDataRows(self,rows, cols, vlayer):
        # get data values from qgis layer's attribute table
        self.signalling.log(2,'GLMClient.getDataRows()',f'')
        try:
            features = vlayer.getFeatures()
            if vlayer.featureCount()==0:
                self.signalling.log(2,'GLMClient.getDataRows()',f'Warning No Features in layer')
                return
            
            self.signalling.log(2,'GLMClient.getDataRows()',f'{vlayer.featureCount()} Features found in layer')

            for feature in features:
                attrs = feature.attributes() # a list containing all the attribute values of this feature
                
                # escape ioGAS illegal chars
                for i in range(len(attrs)):
                    attrs[i] = str(attrs[i]).replace(',','_')  # temp hack replace comars with simple _  as commars breaking the GLM message csv data.
                    #attrs[i] = escape(str(attrs[i])) # 
                
                    # TODO : anything else ? nulls etc 
                
                linestr=','.join(map(str,attrs))  + ',' + self.getCoordinatesFromFeature(feature) # convert attr list to csv string & add coords from fetaure geometry 
                #self.signalling.log(2,'GLMClient.getDataRows()',f'linestr:{linestr}')
                #Use csv line to create new row and add to message's rows collection
                row = gas.xDR()
                row.set_valueOf_(linestr )
                rows.add_DR(row)

        except  Exception as err:
            self.signalling.notify(NotifyLevel.Critical, f'GLMClient.getDataRows : {err.args} ') 

        return 

    # returns a string with csv x,y,z values for given feature; extracted from geometry
    # points can be XY, XYZ, XYZM
    def getCoordinatesFromFeature(self,feature):
        coordStr= ''
        geometry=None
        geometry = feature.geometry()
        if geometry.type() == QgsWkbTypes.Point:  # TODO Test with a 2d point dataset
            coordinate = geometry.asPoint()
            coordStr = f"{coordinate[0]},{coordinate[1]},0"
        elif geometry.type() == QgsWkbTypes.PointGeometry: # seems to be some form of "container" for geoetry - we need to use the get() method to get to the geometry 
            coordStr = (f"{geometry.get().x()},{geometry.get().y()},{geometry.get().z()}")
        else: # unknown geometry type just return placeholder string  : TODO think about this
            coordStr='0,0,0'
        #self.signalling.log(2,'GLMClient.getCoordinatesFromFeature()',f'coordStr:{coordStr}')
        return coordStr


    # set some special columns  from layer properties
    def getSpecialColumns(self, vlayer,cols):

        try:
            # Map projection of - use CRS of the vector layer
            strCRS = vlayer.crs().authid()  # eg EPSG:32613
            
            if "EPSG:" in strCRS:  
                strCRS = strCRS.replace('EPSG:','') # trim to just the EPSG code
            if len(strCRS.strip()) == 0 :
                strCRS='0' # if invalid (blank) indicate non-earth crs to gas (0)
            cols.set_map_epsg(strCRS)
            self.signalling.log(2,'GLMClient.getSpecialColumns()',f'set_map_epsg({strCRS})')
            
            # use our  abstracted coord fields as special cols
            cols.set_map_east('qgis_x')                
            cols.set_map_north('qgis_y')
            cols.set_map_elevation('qgis_z')

        except  Exception as err:
            self.signalling.log(2,'GLMClient.getSpecialColumns()',f'ERROR:{traceback.format_exc()}')
            self.signalling.notify(NotifyLevel.Critical, f'GLMClient.getSpecialColumns : {err.args} ') 

        return 

    # extract Xcolumn definitions from layer
    def getXColumns(self, vlayer,cols):
        
        try:
            for qgis_field in vlayer.fields():
                #self.addXColumn(cols,qgis_field.name(),self.toGLMType(qgis_field.isNumeric()))
                # we kill any gas control fields as they seem to be being retained and screwing up subsequent symbology in the returnd live link data
                # TODO remove the fieldsm, for now we just rename with qgis_prefix
                self.addXColumn(cols,qgis_field.name().replace('ioGAS_Filter__','qgis_' + qgis_field.name()).replace('ioGAS_Shape__','qgis_' + qgis_field.name()).replace('ioGAS_Colour__','qgis_' + qgis_field.name()).replace('ioGAS_Size__','qgis_' + qgis_field.name()), self.toGLMType(qgis_field.isNumeric()))
 
            # we do not know which, if any, of the attribute fields were used to create point geometry, so extract geometry to new fields at end of row
            self.addXColumn(cols,'qgis_x',self.toGLMType(True))
            self.addXColumn(cols,'qgis_y',self.toGLMType(True))
            self.addXColumn(cols,'qgis_z',self.toGLMType(True))


        except  Exception as err:
            self.signalling.notify(NotifyLevel.Critical, f'GLMClient.getXColumns : {err.args} ') 

        return 

    # Create and add a new XColumn       
    def addXColumn(self, cols,fieldName,fieldType):
        try:
            col = gas.xColumn()
            col.set_aliasName(fieldName)
            col.set_origonalName(fieldName)
            col.set_type(fieldType)
            cols.add_column(col)
        except  Exception as err:
            self.signalling.notify(NotifyLevel.Critical, f'GLMClient.addXColumn : {err.args} ') 

        return 

    # Convert qgis data types to gas compatible - gas uses "Numeric", "Text"
    def toGLMType(self, isNumeric):
        
        #TODO - maybe implement specific type conversions -
        if (isNumeric): 
            return "Numeric"
        else :
            return "Text"

    

   
  
  