
import sys
import os
import traceback
import itertools
import operator

from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtCore import *
from qgis.core import *


from ioGAS.gassupport import *
from ioGAS.gaslayerlogic_base import GasLayerLogic_Base
    
class GasLayerLogic_File(GasLayerLogic_Base):
    LEGEND_LIMIT = 60  # >1 if we need to limit number of legend entries - only used where qgis<3.3

    def __init__(self,pSignalling):
        super().__init__(pSignalling)
        self.outputFile = None

    def setOutputFile(self, outputFile):
        self.outputFile = outputFile


    # return true if we are not able to display legend due to qgis version vs number of legend compbos
    # prior to qgis 3.3 (version int = 30300) there is a limit to how large a linear legend can be,
    # caused by an untrappable failure with very large sql statements in sqllite
    # result is that we return false for linear legends where qgis ver < 3.3 and number combos > LEGEND_LIMIT
    # NB: Finding version number may cause exception on earlier versions of qgis; but we really only care whether we have ver < 3.3
    def canDisplayLegend(self):

        if self.legendType!="LINEAR" :   # always display non-Linear Legend
            return True

        # do we have qgis < 3.3
        QGISVersion_LT3_3 = True
        try:
            if Qgis.QGIS_VERSION_INT  >= 30300 : # this will error on qgis versions <3 (?) leaving false
                QGISVersion_LT3_3 = False
        except:
            pass

        if QGISVersion_LT3_3 and len(self.r.uniqueAttCombinations) > self.LEGEND_LIMIT : return False  # can't display
        else: return True
        
    def createGeoPackageLayer(self, outputFile,layerName,gasdata):
        self.gasdata = gasdata
        self.outputFile = outputFile
        layer = self.createGeoPackageFile(layerName)
        return layer

    # convert gasfilereader object to a geopackage file on the filesystem
    def createGeoPackageFile(self,layerName):

        self.signalling.log(1,'GasLayerLogic_File.createGeoPackageFile()',f'Creating Geopackage Layer. File dest : {self.outputFile}')
        
		
        self.removeLayerIfExists(layerName)

# CRS : if epsg not valid then we try to force an invalid CRS
        epsg = self.gasdata.specialColumnEPSG
        try:
            if epsg =='' or epsg is None or epsg=='0':
                crs = QgsCoordinateReferenceSystem() # creates as invalid
                self.signalling.notify(NotifyLevel.Warning,f'The new layer ({layerName}) has an undefined coordinate reference system')
            else:
                crs = QgsCoordinateReferenceSystem('EPSG:' + epsg)
                self.signalling.log(3,'GasLayerLogic_Base.createMemoryLayer()',f'Layer {layerName} created with crs=epsg:{epsg}; crs valid :{crs.isValid()}')
            
        except Exception as ex:
            self.signalling.log(3,'GasLayerLogic_File.createGeoPackageFile()',f'FAILED TO CREATE CRS with epsg=:{epsg}')
            return
            

        # define a re-useable feature template 
        ftr = QgsFeature()
        fields = self.setupFieldStructure()
        ftr.setFields(fields)

        # create a new empty point feature layer at the output file location, with the desired attribute structure
        writer = QgsVectorFileWriter(self.outputFile, 'utf-8', fields, QgsWkbTypes.Point, crs)

        # for each row in source data (ie for each source point)
        for row in self.gasdata.dataAndAtts:

            # add data values to attribute collection
            self.setFeatureAttributeValuesFromRow(row,ftr)

            self.setFeatureGeometryFromRow(row,ftr)

            # add feature to file
            writer.addFeature(ftr)

        del writer # done writing features

        self.signalling.log(2,'GasLayerLogic_File.createGeoPackageFile()',f'Geopackage file created.')

        # create layer from geopackage file and apply the symbology to the layer
        layer = QgsVectorLayer(self.outputFile, layerName, 'ogr')

        self.setLayerSorting(layer)
       
        self.applySymbologyToLayer(layer, True)
        
        self.signalling.log(1,'GasLayerLogic_File.createGeoPackageFile()',f'Layer Created from Geopackage file')

        return layer


