
import sys
import os
import traceback
import itertools
import operator

from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtCore import *
from qgis.core import *

from ioGAS.gassupport import *

# Translation from gas SHapeCodes to a QGIS symbol - a mixed list of built-in symbolMarker definitions, and font charaters
QGIS_SYMBOLS =    [('MARKER', QgsSimpleMarkerSymbolLayerBase.Circle, 0, True),  # 0-8 filled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Square, 0, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 180, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 0, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Diamond, 0, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.SemiCircle, 270, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.SemiCircle, 90, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.HalfSquare, 0, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.HalfSquare, 90, True),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Circle, 0, False),  # 9-17 Unfilled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Square, 0, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 180, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 0, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Diamond, 0, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.SemiCircle, 270, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.SemiCircle, 90, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.HalfSquare, 0, False),
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.HalfSquare, 90, False),

                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Cross2, 90, False),  # X  #18
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Cross, 90, False),  # +  #19

                   # These font based markers apparently not responding to colour in symbology
                   ('FONT','Arial', 'C', 0, False),  #20 C
                   ('FONT','Arial', 'U', 0, False),  # U
                   ('FONT','Arial', '<', 0, False),  # LT <
                   ('FONT','Arial', 'V', 0, False),  # V
                   ('FONT','Arial', 'L', 90, False),  # L rotated gamma
                   ('FONT','Arial', 'L', 0, False),  #25 L  #

                   ('FONT','Arial', 'Z', 0, False),  #26  Z
                   ('FONT','Arial', 'N', 0, False),  #  N
                   ('FONT','Arial', '/', 0, False),  #  /
                   ('FONT','Arial', '\\', 0, False),  #29 \

                   ('FONT','Wingdings', '\xE0', 0, False),  #30  224 r-arrow #30
                   ('FONT','Wingdings', '\xE1', 0, False),  # 225 up-arrow
                   ('FONT','Arial', 'Y', 180, False),  # orig wingdings 229 3 way thingy upside down Y
                   ('FONT','Wingdings', '\xAB', 0, False),  # 171 star
                   ('FONT','Arial', '8', 90, False),  # infinity  8 rotated
                   ('FONT','Arial', '8', 0, False),  #35   char 8
                   ('FONT','Arial', '\xB6', 90, False),  # pilcrow (CR symbol)  8 rotated
                   ('FONT','Arial', '\xB6', 0, False),  #37 pilcrow


                   # ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -45, False),  #38  quadrant triangles38
                   # ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -135, False),
                   # ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -45, True),
                   # ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -135, True),  #41

                    # these 4 shapes changed june 2019 JP
                   ('MARKER', QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 90, False),  # 38  DiagonalHalfSquare pointing Top Left, UNfilled
                   ('MARKER', QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 0, False),  # 39  DiagonalHalfSquare pointing Bottom Left, UNfilled
                   ('MARKER', QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 90, True),  # 40  DiagonalHalfSquare pointing Top Left, Filled
                   ('MARKER', QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 0, True),  # 41  DiagonalHalfSquare pointing Bottom Left, Filled

                   # new symbols june 2019
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -90, False),  #42 left pointing arrowhead unfilled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 90, False),  # right pointing arrowhead unfilled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, -90, True),  # left pointing arrowhead filled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.Triangle, 90, True),  # right pointing arrowhead filled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 180, False),  # DiagonalHalfSquare pointing Top Right, unfilled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, -90, False),  # DiagonalHalfSquare pointing Bottom Right. unfilled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, 180, True),  # DiagonalHalfSquare pointing Top Right, Filled
                   ('MARKER',QgsSimpleMarkerSymbolLayerBase.DiagonalHalfSquare, -90, True)  #49 DiagonalHalfSquare pointing Bottom Right, Filled

                   ]

# Manages layer logic - uses GasData to create a display-ready qgis map layer.
class GasLayerLogic_Base():

    def __init__(self,pSignalling):
        self.signalling = pSignalling
        self.legendType = ''
        self.gasdata = None
        
    def setLegendType(self, legendType):
        self.legendType = legendType   

    # Use Live Link data to Create an in-memory layer
    def createLiveLayer(self,layerName,gasdata):
        self.signalling.log(3,'GasLayerLogic.createLiveLayer()','')
        # any implementation differences req'd?
        
        ##TIMER
        #self.signalling.timer_stamp('createMemoryLayer.start')
        
        layer = self.createMemoryLayer(layerName,gasdata)
        
        ##TIMER
        #self.signalling.timer_stamp('createMemoryLayer.done')
        
        layer.selectionChanged.connect(self.handleSelectionChanged) # wire up the selection changed handler for the layer
        
        return layer


    def handleSelectionChanged(self, selFeatures):

        # We do not use the selFeatures feature IDs supplied by the event. - they are misaligned with data row id's
        # We find the selected features and recover the ROW_ID attribute values and use those instead
        self.signalling.log(3,'GasLayerLogic.handleSelectionChanged()',f'{len(selFeatures)} Selected Features')
        layer = self.findLiveLayer()
        selection = layer.selectedFeatures()

        selFeatureRowIDs = []
        for ftr in selection:
            selFeatureRowIDs.append(ftr[LIVELINK_ROWIDFIELDNAME])
            
        self.signalling.raise_selection_changed(selFeatureRowIDs)


    # Create an in-memory layer (aka "Scratch" layer)
    def createMemoryLayer(self,layerName,gasdata):
        
        self.signalling.log(1,'GasLayerLogic_Base.createMemoryLayer()',f'')
        self.gasdata = gasdata

        attrErrorCnt = 0
        geomErrorCnt = 0

        # coord sys
        epsg = self.gasdata.specialColumnEPSG
        # crs = QgsCoordinateReferenceSystem('EPSG:' + epsg)   # currently we explicitly set mem layer crs  using epsg code -

        # define a feature - a kind of template
        ftr = QgsFeature()
        fields = self.setupFieldStructure()
        ftr.setFields(fields)

        # create empty layer with correct feature type and CRS
        layer_uri=''
        try:
            if epsg =='' or epsg is None or epsg=='0':
                layer_uri="Point"  # no epsg (nonearth) - create layer without setting CRS
                layer = QgsVectorLayer(layer_uri, layerName, "memory")
                # try to force layer to an invalid CRS
                no_crs = QgsCoordinateReferenceSystem() # creates as invalid
                layer.setCrs(no_crs) 
                self.signalling.notify(NotifyLevel.Warning,f'The new layer ({layerName}) has an undefined coordinate reference system')
            else:
                layer_uri=f"Point?crs=EPSG:{epsg}"  # we have valid epsg
                layer = QgsVectorLayer(layer_uri, layerName, "memory")
            
            self.signalling.log(3,'GasLayerLogic_Base.createMemoryLayer()',f'Layer {layerName} created with uri=:{layer_uri}; layer crs valid :{layer.crs().isValid() }')
       
        except Exception as ex:
            self.signalling.log(3,'GasLayerLogic_Base.createMemoryLayer()',f'FAILED TO CREATE Layer {layerName}, with uri=:{layer_uri}, Aborting\n{traceback.format_exc()}')
            return
        
        self.setLayerSorting(layer)

        # configure data provider for layer with attribute structure
        provider = layer.dataProvider()
        provider.addAttributes(fields)

        # changes are only possible when editing the layer
        layer.startEditing()
        
        featCnt=0
        for row in self.gasdata.dataAndAtts:    # for each row in source data (ie for each source point)
            
            # add data values to attribute collection
            if not self.setFeatureAttributeValuesFromRow(row,ftr):
                attrErrorCnt += 1 # if we had attribute error for this feature, we flag but ignore error and continue processing the row

            if not self.setFeatureGeometryFromRow(row, ftr):
                geomErrorCnt += 1
                continue  # if we can't make geometry, skip to next row

            provider.addFeatures([ftr]) # add feature to layer via it's data provider
            featCnt +=1
            
        layer.commitChanges() # stop editing and save features to layer

        layer.updateExtents() # update layer's spatial extent after new features have been added

        self.applySymbologyToLayer(layer,False)

        self.signalling.log(3,'GasLayerLogic_Base.createMemoryLayer()',f'{featCnt} features added to Layer {layerName} ')

        errMsg = ''
        if attrErrorCnt>0 :
            errMsg = f"{attrErrorCnt} Attribute error(s) occurred while creating GIS features\r\n"
        if geomErrorCnt>0 :
            errMsg = f"{geomErrorCnt} Geometry (coordinate) errors occurred while creating GIS features\r\n"
        if len(errMsg) > 0 :   
            self.signalling.log(3,'GasLayerLogic_Base.createMemoryLayer()',f'ERROR:{errMsg}')
            
        return layer

# apply the symbology to a layer
    def applySymbologyToLayer(self,layer,isGeoPackage):
        rule = None
        # if self.legendType == 'HIERARCHICAL':
        #     rule = self.generateRulesHierarchical()
        # else:
        #     rule = self.generateRulesLinear()

        rule = self.generateRulesLinear()
        renderer = QgsRuleBasedRenderer(rule)
        layer.setRenderer(renderer)
        if isGeoPackage: layer.saveStyleToDatabase('main', 'ioGAS', True, '')  # for gpkg we save the style into the gpkg file

    # Set feature attribute values from data row
    def setFeatureAttributeValuesFromRow(self,row,ftr):
    
        # QGIS Version issue: wrt non-numeric values in numeriic fields. Gas allows non-numeric values in numeric fields.
        # In QGIS <= 3.10 non-numeric values would be ignored by qgis, but the geometry would be created regardless.
        # But from ver 3.16 QGIS this behaviour changed : if non-numerics were present in numeric fields then the geometry would fail (silently) to be created.
        # Thus all data loaders must fix this eg by replacing non numerics with null, 0 etc
    
        try :
            cellValue=None
            attributes = []
            for cellValue in row:

                if  cellValue== None: # add nulls directly
                    attributes.append(None)
                else:
                    attributes.append(cellValue) # no need to cast - already done previously

            #self.signalling.log(3,'GasLayerLogic_Base.setFeatureAttributeValuesFromRow()',f'attributes: {attributes}')
            ftr.setAttributes(attributes)  # apply to feature

        except Exception as ex:
            self.signalling.log(3,'GasLayerLogic_Base.setFeatureAttributeValuesFromRow()',f'ERROR: {feat_rowid}\n{traceback.format_exc()}')
            return False

        return True

    # Add required empty column structure to hold attribute values that will be applied to each created point feature
    def setupFieldStructure(self):
        fields = QgsFields()
        for i, c in enumerate(self.gasdata.columns):
            dataType = QVariant.String
            if c.type == "Text":
                dataType = QVariant.String
            if c.type == "Numeric":
                dataType = QVariant.Double
                
            #self.signalling.log(3,'GasLayerLogic_Base.setupFieldStructure()',f' c.name, dataType:{c.name},{dataType}')
            f = QgsField(c.name, dataType)
            fields.append(f)

        # gas special attribute fields - we need these as they drive the legend via the legend rules
        fields.append(QgsField(self.gasdata.GAS_FILTER, QVariant.Int))  # TODO Think we can remove this as already filtered for  visible rows only?
        fields.append(QgsField(self.gasdata.GAS_COLOUR, QVariant.Int))  # TODO - hou;d change to numeric - can we make them ints or zero d.p. ?
        fields.append(QgsField(self.gasdata.GAS_SHAPE, QVariant.Int))
        fields.append(QgsField(self.gasdata.GAS_SIZE, QVariant.Int))

        strlist = ''
        for field in fields:
            strlist += field.name()
        #self.signalling.log(3,'GasLayerLogic_Base.setupFieldStructure()',f'fields: {strlist}')

        return fields


     # set feature geometry/location from coord values in the data row
    
    def setFeatureGeometryFromRow(self, row, ftr):
        ftr.setGeometry(None) #  discard previous (ftr obj is recycled as template)
        if self.gasdata.indexZ == -1:  # 2D
            try:
                x = float(row[self.gasdata.indexX])  # coords
                y = float(row[self.gasdata.indexY])
                ftr.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(x, y)))
            except:
                ftr.setGeometry(None)
                #self.signalling.log(2,'GasLayerLogic_Base.setFeatureGeometryFromRow()',f'row with bad x,y')
                return False

        else:  # 3D
            try:
                x = float(row[self.gasdata.indexX])  # coords
                y = float(row[self.gasdata.indexY])
                z = float(row[self.gasdata.indexZ])
                ftr.setGeometry(QgsGeometry(QgsPoint(x, y, z)))
            except:
                try :
                    ftr.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(x, y)))  # we try to create a 2D feature instead
                    #self.signalling.log(2,'GasLayerLogic_Base.setFeatureGeometryFromRow()',f'3D Row was created as 2D feature instead')
                except:
                    ftr.setGeometry(None)
                    #self.signalling.log(2,'GasLayerLogic_Base.setFeatureGeometryFromRow()',f'Unexpected Geometry error:{ sys.exc_info()[0]}')
                    return False
        return True
        
    #  Set symbol shape - look up a shape from our qgis shapes list and set for this symbol        
    def SetSymbolShapeAndColour(self, symbol, shape, colour):

        try:
            # TODO : If symbol index is outside of dictionary range, display a black ! (bang)
            qgisSymbolRef = QGIS_SYMBOLS[int(shape.shapeCode) % len(QGIS_SYMBOLS)]    # a tuple from our symbol Library  MAP_SYMBOLS, wrapping where index higher than number of items in list
            
            # resolve colour
            try:
                tupcol = RGBAfromInt(int(colour.colour)) # convert int from gas to components - r,g,b,a
                #self.signalling.log(3,'GasLayerLogic_Base.SetSymbolShapeAndColour()',f'DEBUG: colour:{colour}=r,g,b,a={tupcol[0]},{tupcol[1]},{tupcol[2]},{tupcol[3]}')
                rgbaQcolour = QColor.fromRgb(tupcol[0],tupcol[1],tupcol[2])  # convert to QColor
            except Exception as ex:
                self.signalling.log(3,'GasLayerLogic_Base.SetSymbolShapeAndColour()',f'ERROR: Failed to set colour:{colour}\n{traceback.format_exc()}')
                rgbaQcolour = QColor.fromRgb(0, 0, 0)  # fall back to black

            # resolve shape
            if qgisSymbolRef[0] == 'FONT':  # font-based  ('FONT','Arial', 'C', 0, False),  # C  #20
                symbol.changeSymbolLayer(0, QgsFontMarkerSymbolLayer(qgisSymbolRef[1], qgisSymbolRef[2]))
                symbol.setAngle(qgisSymbolRef[3])
                symbol.symbolLayer(0).setColor(rgbaQcolour)
                # if qgisSymbolRef[4] != True:  # unfilled
                #     rgbaQcolourFill = QColor(0, 0, 0, 0)  # transparent
                #     symbol.symbolLayer(0).setColor(rgbaQcolourFill)
            else:  # Marker eg ('MARKER',QgsSimpleMarkerSymbolLayerBase.Circle, 0, True)
                symbol.symbolLayer(0).setShape(qgisSymbolRef[1])
                symbol.setAngle(qgisSymbolRef[2])
                symbol.symbolLayer(0).setStrokeColor(rgbaQcolour)
                if qgisSymbolRef[3] != True:  # unfilled
                    rgbaTransparent = QColor(0, 0, 0, 0)  # transparent
                    symbol.symbolLayer(0).setFillColor(rgbaTransparent)
                else:
                    symbol.symbolLayer(0).setFillColor(rgbaQcolour)
        
        except Exception as ex:
            self.signalling.log(3,'GasLayerLogic_Base.SetSymbolShapeAndColour()',f'ERROR: symbol:{symbol}; shape:{shape}; colour:{colour}\n{traceback.format_exc()}')
        
        return


    # Set symbol size from from gas lookup table entry and apply to symbol
    def SetSymbolSize(self, symbol, size):
        pointSize = float(size.size) / 3.0  # hard coded scaling factor
        symbol.setSize(pointSize)
        return

    # Attempt to force attribute table viewer rows to be sorted in feature added order, for a layer
    def setLayerSorting(self, layer):
        config = layer.attributeTableConfig()
        config.setSortExpression('$id')  # inbuilt feature id ascends in order they were added
        config.setSortOrder(Qt.AscendingOrder)
        layer.setAttributeTableConfig(config)


# we base the symbology on the gas control field values, rather than the cell values
    def generateRulesLinear(self):
        ruleTemplate = '"{field1}" = \'{value1}\' and  "{field2}" = \'{value2}\' and  "{field3}" = \'{value3}\'  '
        rootRule = QgsRuleBasedRenderer.Rule(None)

        for uaIndex, ua in enumerate(self.gasdata.uniqueAttCombinations):
            #indices into shape/colour/size atts for this unique combination
            colourAtt = int(ua[0])
            if colourAtt >= len(self.gasdata.colours):
                self.log(3,'GasLayerLogic_Base.generateRulesLinear()',f'Colour lookup problem index={colourAtt}')
            colour = self.gasdata.colours[colourAtt]  # entry from gas metadata colour lookup table
            shapeAtt = int(ua[1])
            shape = self.gasdata.shapes[shapeAtt] # entry from gas metadata shape lookup table
            sizeAtt = int(ua[2])
            size = self.gasdata.sizes[sizeAtt] # entry from gas metadata size lookup table

            #Create a symbol and related rule the rule
            symbol = QgsSymbol.defaultSymbol(QgsWkbTypes.PointGeometry).clone()
            rule = QgsRuleBasedRenderer.Rule(symbol)

            # Style the symbol
            qgisSymbolRef = QGIS_SYMBOLS[int(shape.shapeCode) % len(   QGIS_SYMBOLS)]  # a tuple from our symbol Library  MAP_SYMBOLS, wrapping where index higher than number of items in list

            self.SetSymbolShapeAndColour(symbol, shape, colour)

            self.SetSymbolSize(symbol, size)

            # Configure the rule - we get the above symbol if the rule matches (ie if the rows shape/colour/size att values all match)
            rule.setFilterExpression(ruleTemplate.format(field1=self.gasdata.GAS_COLOUR, value1=colourAtt, field2=self.gasdata.GAS_SHAPE, value2=shapeAtt, field3=self.gasdata.GAS_SIZE, value3=sizeAtt))
            rule.setLabel(colour.name + ", " + shape.name + ", " + size.name)
            #self.signalling.log(3,'GasLayerLogic_Base.generateRulesLinear()',colour.name + ", " + shape.name + ", " + size.name)
            
            rule.setDescription("ioGAS rule [" + str(uaIndex) + "] " + str(ua))

            rootRule.appendChild(rule)

        return rootRule
        
    # Remove any map layer of same name
    # NB, we are not deleting files, just removing from the map's layers collection
    def removeLayerIfExists(self, layerName):
        layersall = QgsProject.instance().mapLayers()
        layers = QgsProject.instance().mapLayersByName(layerName)
        if len(layers) > 0:
            for layer in layers:
                self.signalling.log(3,'GasLayerLogic.removeLayerIfExists()',f"Removing previous layer with this name ({layerName}).")
                QgsProject.instance().removeMapLayers([layer.id()])  


    def removeLiveLayerIfExists(self):
        livelayer = self.findLiveLayer()
        if not livelayer == None :
            self.signalling.log(3,'GasLayerLogic.removeLayerIfExists()',f"Removing previous Live layer with this name ({livelayer.name()}).")
            QgsProject.instance().removeMapLayers([livelayer.id()])  

    
    def findLiveLayer(self):
        layer=None
        layers = QgsProject.instance().mapLayers().values()
        for lyr in layers:
            if lyr.name().endswith(LIVELINK_LAYERNAME):
                layer = lyr
                break
        return layer


# does this layer use rules based labelling
    def hasRuleBasedLabeling(self, layer):
        label_settings = layer.labeling()
        if label_settings:
            if label_settings.type() == "rule-based":
                return True
        return False

  
    # returns a copy of the labelling rules collection from source_layer
    def copyRuleBasedLabeling(self,source_layer):
        source_labeling = source_layer.labeling()  # QgsRuleBasedLabeling
        rootRuleCopy = None
        
        if source_labeling and source_labeling.type() == 'rule-based':
            sourceRoot = source_labeling.rootRule()  # QgsRuleBasedLabeling.Rule
            rootRuleCopy = QgsRuleBasedLabeling.Rule(QgsPalLayerSettings())   # create a new rootRule, QgsRuleBasedLabeling.Rule

            # Iterate the rules and add to new root
            for rule in source_labeling.rootRule().children():    # source_labeling.rootRule() returns exist ruleroot from source layer, QgsRuleBasedLabeling.Rule
                rootRuleCopy.appendChild(rule.clone())

            return rootRuleCopy


    def applyLabellingRules(self,target_layer, RulesRoot):
            #Apply label configuration in RulesRoot to a target layer
            ## root is QgsRuleBasedLabeling.Rule
            target_labeling = target_layer.labeling() # QgsRuleBasedLabeling
            rules = QgsRuleBasedLabeling(RulesRoot)
            target_layer.setLabeling(rules)
            target_layer.setLabelsEnabled(True)
        

    def unused_duplicate_layer_as_temporary(self, sourceLayer):
        # Create a copy of the sourceLayer and return as a temporary layer , not added to the project
        layer_copy = sourceLayer.clone()
        temporary_layer_name = "temp_01" 
        layer_copy.setName(temporary_layer_name)        
        return layer_copy


    def refresh_layers(self,layers):
        for layer in layers:
            layer.triggerRepaint()

