
import sys
import os
import traceback

from collections import namedtuple
from operator import itemgetter


try:
    import ioGAS.gaslink as gaslink
    import ioGAS.gas as gas  # used by gaslink
    import ioGAS.gassupport as gassupport
    from ioGAS.gasdata_base import *

except:
    import gaslink
    import gas
    #import gassupport 
    from gasdata_base import *
    from gassupport import *

# Can load data from GasLink (Live link)
class GasData_Link(GasData_Base):
    # most recent GLM messaage data, nb both must be present before we can unpack and present data to applications
    data_message = None
    attr_message = None
        
    def __init__(self, pSignalling,pClientType):
        super().__init__(pSignalling)
        self.clientType = pClientType
        #self.allcolumns = {}  

    def set_attr_message(self, oMsg): 
        self.attr_message = oMsg 

        
    def set_data_message(self, oMsg):
        self.data_message = oMsg 


    def load_data(self):       
        #unpack the data and attribute messages into a gasData memory structure
        # self.signalling.log(3,'GasData_Link.load_data()','Loading...')
 
        # Only proceed if we have both valid data and attributes
        if (self.data_message==None or self.attr_message==None or self.nullMessage==True) :
            self.warning = False # not an error, just can't load without both
            return False

        #self.log(3,'GasData_Link.load_data()',f'attr_message:{convert(self.attr_message)}')
        #self.log(3,'GasData_Link.load_data()',f'data_message:{convert(self.data_message)}')

        try :
            # resolve table/filename from GAS       
            fileName = self.data_message.get_name() # filename from gas
            fileName = fileName.replace(f'{self.clientType}: ','').replace('.gas','')  # remove qlm client prefix and .gas filename extension
            self.name = fileName 

            ### 1. read metadata
            self.load_attributeTables()

            self.symbologyTitle = self.colourTitle + ", " + self.shapeTitle + ", " + self.sizeTitle

            self.load_allColumns() 


            # add an explicit RowID column as last column        
            col = self.Column(gassupport.LIVELINK_ROWIDFIELDNAME, 'Text', len(self.allcolumns)) #JP Changed to Text else QGIS3.16.6 reports as decimal and selection fails
    
            self.allcolumns[col.name] = col
            self.columns.append(col)
            
                
            #self.signalling.log(3,'GasData_Link.load_data()',f' columns:{self.columns}')
            self.load_SpecialColumns()    

            # [Link data has no selected columns / variable history mechanism]
            
            ### 2.  read data and atts from csv
            self.load_DataAndAtts()   # combine data with attributes
            
            self.extractUniqueLegendCombinations()

            self.signalling.log(3,'GasData_Link.load_data()','all done...')

        except Exception as ex:
            self.signalling.log(3,'GasData_Link.load_data()',f'FAILED to load data from data message\nERROR:\n{traceback.format_exc()}')
            #self.signalling.log(3,'GasData_Link.load_data()',f'DATA MESSAGE \n:{convert(self.data_message)}')     
 
        return True


    def load_attributeTables(self):
         # we convert all of the attribute look up tables to lists of named tuples
        try: 
            self.filters = []
            filtersWrapper = self.attr_message.get_filterAttributes().get_filterAttribute()
            for ca in filtersWrapper:
                filter = self.Filter( ca.get_name(),  ca.get_visible() )
                self.filters.append(filter)
                
            self.colours = []
            self.colourTitle = self.attr_message.get_colourAttributes().get_title().strip()
            coloursWrapper = self.attr_message.get_colourAttributes().get_colourAttribute()
            for ca in coloursWrapper:
                colour = self.Colour( ca.get_name(), ca.get_colour(), ca.get_visible() )
                self.colours.append(colour)
            

            self.shapes = []
            self.shapeTitle = self.attr_message.get_shapeAttributes().get_title().strip()
            shapesWrapper = self.attr_message.get_shapeAttributes().get_shapeAttribute()
            for ca in shapesWrapper:
                shape = self.Shape( ca.get_name(), ca.get_shape(), ca.get_visible() )
                self.shapes.append(shape)

            self.sizes = []
            self.sizeTitle = self.attr_message.get_sizeAttributes().get_title().strip()
            sizesWrapper = self.attr_message.get_sizeAttributes().get_sizeAttribute()
            for ca in sizesWrapper:
                size = self.Size( ca.get_name(), ca.get_size(), ca.get_visible() )
                self.sizes.append(size)
        
        except Exception as ex:
            self.signalling.log(3,'GasData_Link.load_attributeTables()',f'FAILED to get Attribute Tables from data message:\n:{convert(self.data_message)}')

    def load_allColumns(self):
        
        try :
            index=0
            for xcol in  self.data_message.get_columns().get_column():  # xcol is an xColumn 
                name = xcol.get_aliasName() # self.unescape not required ?
                col = self.Column(name, xcol.get_type(), index)
                self.allcolumns[name] = col
                self.columns.append(col)
                index = index +1
        except Exception as ex:
             self.signalling.log(3,'GasData_Link.load_allColumns()',f'FAILED to load columns from data message\nERROR:\n{traceback.format_exc()}')

    def load_SpecialColumns(self):
        #Get special columns - these are specific properties of specialColumns property - not a collection
        try:
            specialColumns = self.data_message.get_specialColumns()

            #self.signalling.log(3,'GasData_Link.load_data()',f'DEBUG : self.columns : {str(self.columns)}')  
            #self.signalling.log(3,'GasData_Link.load_data()',f'DEBUG : specialColumns : {str(specialColumns.__dict__)}') 

            if hasattr(specialColumns,'id'):
                self.specialColumnID = specialColumns.id 
            
        

            #must have E and N, but elevation is optional        
            try:
                self.indexX = -1
                if hasattr(specialColumns,'map_east'):
                    self.specialColumnEast = specialColumns.map_east
                    self.indexX = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnEast)][0] # track the index of x column
                
                self.indexY = -1
                if hasattr(specialColumns,'map_north'):
                    self.specialColumnNorth = specialColumns.map_north
                    self.indexY = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnNorth)][0] # track the index of y column
            except Exception as ex:
                self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'specialColumns.E or N \nERROR:\n{traceback.format_exc()}') 
                raise (ValueError(ERR_EN))


            self.indexZ = -1
            if hasattr(specialColumns,'map_elevation'):
                self.specialColumnElevation = specialColumns.map_elevation
                if self.specialColumnElevation != None and self.specialColumnElevation:
                    self.indexZ = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnElevation)][0] # track the index of z column
                else:
                    self.indexZ = -1

            # Projection
            try:   
                if hasattr(specialColumns,'map_epsg'): 
                    self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'specialColumns.map_epsg found in message:{specialColumns.map_epsg}')  
                    self.specialColumnEPSG = specialColumns.map_epsg
                    if self.specialColumnEPSG is None :
                        self.specialColumnEPSG = '0'   

                else:
                    self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'No specialColumns.map_epsg NOT found in message, setting to 0') 
                    self.specialColumnEPSG = '0' 
                    
                self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'Setting Projection from map_epsg to:{self.specialColumnEPSG}')    
            except Exception as ex:
                self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'ERROR Setting Projection from map_epsg.') 
            
        except Exception as ex:
             self.signalling.log(3,'GasData_Link.load_SpecialColumns()',f'FAILED to load columns from data message\nERROR:\n{traceback.format_exc()}')
    
    def load_DataAndAtts(self):
         # combine data with attributes from attribute message and if visible, add to our data, atts and dataandatts collections 
            # For link data we get all fields, there is not selected columns metadata
        try:
            rowList = self.data_message.get_DRs().get_DR()
            self.rows = []
            self.viz = [None] * len(rowList)
            
            rowid=0 # we keep a manual rowID to match the provided data - allows us to do send/apply attributes reliably
            for i in range(0, len(rowList)):
                row = self.data_message.get_DRs().get_DR()[i].get_valueOf_().split(',')  #list
            
                # add the row_id value, as a string to be consistent with incoming
                row.append(str(rowid))
                rowid +=1
                
                tup = tuple(row)  

                #self.signalling.log(1,'', f'tup:{str(tup)}\ncolumns:{str(self.columns)}')

                data = gassupport.lookup(tup, self.columns) # also repairs text in numerics

                rowAtts = self.attr_message.get_rows().get_Row()[i]  # xRow
                #self.signalling.log(3,'GasData_Link.load_data()',f'rowAtts:{rowAtts}')    
                att=(str(rowAtts.filter),str(rowAtts.colour), str(rowAtts.shape), str(rowAtts.size))
                symbolicAtt=(str(rowAtts.colour), str(rowAtts.shape), str(rowAtts.size))
                #self.signalling.log(3,'GasData_Link.load_data()',f'symbolicAtt:{symbolicAtt}')    
                dataAndAtt = data + att
                #self.signalling.log(3,'GasData_Link.load_data()',f'dataAndAtt:{dataAndAtt}')    
                # only add rows that should be visible to our row collection
                if self.rowVisible(rowAtts, self.colours, self.shapes, self.sizes, self.filters):
                    self.viz[i] = True               
                    
                    self.data.append(data)
                    self.atts.append(att)
                    self.symbolicAtts.append(symbolicAtt)
                    self.dataAndAtts.append(dataAndAtt)

                else:
                    self.viz[i] = False

        except Exception as ex:
            self.signalling.log(3,'GasData_Link.load_DataAndAtts()',f'FAILED to load data from data message\nERROR:\n{traceback.format_exc()}')
 

        
