# A GasData that can load a gas file
# No dependancies

import os
import traceback
import zipfile
import csv
import shutil
import tempfile
from collections import namedtuple
from operator import itemgetter
from pathlib import Path

# METHOD ONE
 #  - hacky but  works!
try:
    from ioGAS.gasdata_base import *
    from ioGAS.gassupport import *
except:
    from gasdata_base import *
    from gassupport import *

# METHOD TWO  MOVE  test file up a level
# from ioGAS.gasdata import *
# - would be ok but it imports the __init__ and you get
#  File "C:\work\gas\uber\qgis\ioGAS\__init__.py", line 4, in <module>
#     from ioGAS.qgisplugingas import QGisPluginGas
#   File "C:\work\gas\uber\qgis\ioGAS\qgisplugingas.py", line 7, in <module>
#     from qgis.core import *
# ModuleNotFoundError: No module named 'qgis'

# METHOD THREE
# from .gasdata import * #
# - does not work  'attempted relative import with no known parent package' when running tests



# UI and version strings
REQUIRED_VERSION = "6.3"

#todo
#pylint?
#remove? duplicated geocheem data table (waste of mem?)

       
class GasData_File(GasData_Base):

 

    def __init__(self, gasfilepath, signalling):
        super().__init__(signalling)

        self.gasfile = gasfilepath


        ### Public fields (the non list declarations here are for doco only!)

        self.versionString = ""       
        self.version = 0.0  # numeric (float) version of loaded gas file as x.y float

        self.warning = ""

 

# appends, ignoring duplicates (returns false)
    def safeColumnAppend(self,cols, col):
        n = col.name
        if (containsName(cols, n)):
            ret = False
        else:
            col = self.Column(n, col.type, col.index)
            cols.append(col)
            ret = True
        return ret

# Load data from gas file unto memory structure
    def load_data(self):  
        try:
            #unzip
            tempDir = tempfile.mkdtemp() #cleaned up at end
            self.signalling.log(2, 'GasData_File.load_data()', f'loading {self.gasfile} using {tempDir}')

            zip_ref = zipfile.ZipFile(self.gasfile, 'r')
            zip_ref.extractall(tempDir)
            zip_ref.close()
            dataFile = os.path.join(tempDir ,"data.csv")
            metadataFile = os.path.join(tempDir ,"metadata.1.xml")
            if not Path(metadataFile).is_file(): # use newer format (metadata.1.xml) if present
                metadataFile = os.path.join(tempDir ,"metadata.xml") # use older format
            versionFile = os.path.join(tempDir, "version.txt")

            ### 0. read version, and check it
            p = Path(versionFile)
            if p.is_file():
                f = open(versionFile)
                verText = list(csv.reader(f))[0][0] #single cell - the version like x.y else other non std alphnumerics inc dev, test, alpha etc
                f.close()
                
            else: # could not find/open ver file this can happen with certain versions from a period around 7.1? we optimistically assume that file will be useable
                verText = "Unknown Version, ensure > v5"

            if not self.checkFormatVersion(verText):
                return False


            ### 1. read metadata
            import xml.etree.ElementTree as ET
            doc = ET.parse(metadataFile).getroot() #nb encoding is set in the file, so no need to reset

            self.load_attributeTables(doc)

            self.symbologyTitle = self.colourTitle + ", " + self.shapeTitle + ", " + self.sizeTitle

            # For file loading - we select all of the metadata columns into self.allColumns, then
            # we subset the selected columns and symbology columns into self.columns

            self.load_allColumns(doc) 
            # So by now self.Columns has all of the Special columns + any selected columns

            self.load_SpecialColumns(doc)

            #also add the selected columns, in order
            self.load_SelectedColumns(doc)

            # So by now self.Columns has all of the Special columns + any selected columns

            # NB the self.columns may be not in data column order.
            # it is convenient to have a column index orderd list that matches the csv data fields
            columns_indexorder = sorted(self.columns,key = lambda x: x[2])   # sortg by the column index property (3rd element in tuple)
            
            #self.signalling.log(2, 'GasData_File.load_data()', f'columns_indexorder type :{type(columns_indexorder)}\ncols {columns_indexorder}')
            
            ### 2.  read data and atts from csv
            self.load_DataAndAtts(dataFile)  # combine data with attributes

            self.extractUniqueLegendCombinations()
            
            self.signalling.log(3,'GasData_File.load_data()','all done...')

        except WrongGASVersionException as ex:
            self.validationMessage = f'{ex}'
            self.signalling.log(3,'GasData_File.load_data()',self.validationMessage)
            return False
        except MissingCoordColumnException as ex:
            self.validationMessage = f'MISSING COORDS SPECIAL COL'
            self.signalling.log(3,'GasData_File.load_data()',self.validationMessage)
            return False
            
        except Exception as ex:
            self.validationMessage = f'FAILED to load data from data message\nERROR:\n{traceback.format_exc()}'
            self.signalling.log(3,'GasData_File.load_data()',self.validationMessage)
            #self.signalling.log(3,'GasData_Link.load_data()',f'DATA MESSAGE \n:{convert(self.data_message)}')     
            return False
        else:
            return True

        finally: # always runs - exception or not
            #clean up
            shutil.rmtree(tempDir)

    # new format checking logic; returns true of acceptable 
    def checkFormatVersion(self, versionText):

        try:
            self.versionString = versionText
            # first we check for non numeric, if so accept
            if not versionText.replace('.','',1).isdigit() :  # non-numeric, ie not x.y, incl x.y.z is non-numeric
                self.version = self.intVer(REQUIRED_VERSION)/1000 # we set ver notionally to min reqd
                self.validationMessage = f'Accepting non-numeric GAS file version : {versionText}'
                return True

            # is numeric, check  > min ver required
            ver = self.intVer(versionText)
            if ver < self.intVer(REQUIRED_VERSION):
                self.version = ver/1000
                self.validationMessage = f"File has version {versionText}, but must have been saved with ioGAS version {REQUIRED_VERSION} or later"
                return False

            # >= reqd ver    
            self.version = ver/1000
            self.signalling.log(2, 'GasData_File.checkFormatVersion()', f'Accepting GAS file version {self.version}')
            return True

        except Exception as ex:
            self.validationMessage = f'FAILED to interpret version from GAS file\nERROR:\n{traceback.format_exc()}'  
            return False

# extract numeric version from string, [ as int ver is * 1000, eg 6.3 -> 6300]
    def intVer(self,ver):
        try:
            vers = ver.split(".")
            ret = int(vers[0]) * 1000 + int(vers[1]) * 100
        except:
            ret = -1000 # is ver -1
        return ret



    def load_attributeTables(self,doc):

        fas = doc.find("extraAttributes2")
        for fa in fas.findall("extraAttribute"):
            filter = self.Filter(unescape(fa.find("name").text),
                            fa.find("visible").text == 'true')
            self.filters.append(filter)

        cas = doc.find("colourAttributes")
        self.colourTitle = unescape(cas.find("title").text).strip()
        for ca in cas.findall("colourAttribute"):
            colour = self.Colour(unescape(ca.find("name").text),
                            ca.find("colour").text,
                            ca.find("visible").text == 'true')
            self.colours.append(colour)

        sas = doc.find("shapeAttributes")
        self.shapeTitle = unescape(sas.find("title").text).strip()
        for sa in sas.findall("shapeAttribute"):
            shape = self.Shape(unescape(sa.find("name").text),
                          sa.find("shapeCode").text,
                          sa.find("visible").text == 'true')
            self.shapes.append(shape)

        zas = doc.find("sizeAttributes")
        self.sizeTitle = unescape(zas.find("title").text).strip()
        for za in zas.findall("sizeAttribute"):
            size = self.Size(unescape(za.find("name").text),
                        za.find("size").text,
                        za.find("visible").text == 'true')
            self.sizes.append(size)


    def load_allColumns(self, doc):
        try :
            self.allColumns = {}
            cols = doc.find("columns")
            index=0
            for col in cols.findall("column"):
                name = unescape(col.find("aliasName").text)
                c = self.Column(name, col.find("type").text, index)
                self.allColumns[name] = c
                index = index +1
        except Exception as ex:
             self.signalling.log(3,'GasData_File.load_allColumns()',f'FAILED to load columns from data message\nERROR:\n{traceback.format_exc()}')


    def load_SelectedColumns(self, doc):
        
        vshist = doc.find("variableSelectionHistory")
        # self.debugmsg = self.debugmsg + ET.tostring(vshist, encoding='utf8').decode('utf8') + ","
        vsh = vshist.findall("variableSelection")
        if len(vsh) > 0:
            vs = vsh[-1] #last
            for sel in vs.findall("variable"):
                name = unescape(sel.find("variableName").text)
                #self.debugmsg = self.debugmsg + name + ","
                column = self.allColumns[name]
                
                #self.selCols.append(column)
                if (column not in self.columns):
                    # prevent adding special cols twice because we only store a single index of col pos
                    self.columns.append(column)


    def load_SpecialColumns(self, doc):
        try:
            # optional metadata
            sc = doc.find("specialColumns")
            # nb special cols always exported, even if not selected

            # ID filed is optional
            try:
                self.specialColumnID = unescape(sc.find("id").text)
                self.safeColumnAppend(self.columns, (self.allColumns[self.specialColumnID]))
            except:
                pass

            # Map_East/North are mandatory : dont load file if not present
            try:
                self.specialColumnEast = unescape(sc.find("map_east").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnEast]) #always export specials
                self.indexX = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnEast)][0]
            except:
                raise (MissingCoordColumnException(ERR_EN))

            try:
                self.specialColumnNorth = unescape(sc.find("map_north").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnNorth])  # always export specials
                self.indexY = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnNorth)][0]
            except:
                raise (MissingCoordColumnException(ERR_EN))

            # Lat / Long optional
            try:
                self.specialColumnLongitude = unescape(sc.find("wgs84longitude").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnLongitude])
            except:
                pass

            try:
                self.specialColumnLatitude = unescape(sc.find("wgs84latitude").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnLatitude])
            except:
                pass

            # datetime optional
            try:
                self.specialColumnDateTime = unescape(sc.find("dateTime").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnDateTime])
            except:
                pass

            # elevation is optional - but if present we want to create 3d geometry
            try:
                self.specialColumnElevation = unescape(sc.find("map_elevation").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnElevation])  # always export specials
                if self.specialColumnElevation != None and self.specialColumnElevation:
                    self.indexZ = [i for i, c in enumerate(self.columns) if (c.name == self.specialColumnElevation)][0]
                else:
                    self.indexZ = -1
            except:
                self.indexZ = -1   # valid to have no elevation column, but we flag as missing

            # Projection
            # if map_epsg is not present we will treat as non-earth . map_epsg supercedes map_projection col
            try:
                self.specialColumnEPSG = unescape(sc.find("map_epsg").text)
                self.signalling.log(2, 'GasData_File.load_SpecialColumns()', f'EPSG read as {self.specialColumnEPSG}')
            except:
                self.signalling.log(2, 'GasData_File.load_SpecialColumns()', f'EPSG not found (assuming non earth)')
                pass

            # these ones all from dh and line plots
            try:
                self.specialColumnGroup = unescape(sc.find("group").text)
                self.safeColumnAppend(self.columns, self.allColumns[self.specialColumnGroup])
            except:
                pass

            # Drillhole stuff optional
            dh = doc.find("DHOptions")

            try:
                self.specialColumnFrom = unescape(dh.find("fromField").text)
                self.safeColumnAppend(self.columns, (self.allColumns[self.specialColumnFrom]))
            except:
                pass

            try:
                self.specialColumnTo = unescape(dh.find("toField").text)
                self.safeColumnAppend(self.columns, (self.allColumns[self.specialColumnTo]))
            except:
                pass

        except WrongGASVersionException as ex:
            raise(ex)

        except MissingCoordColumnException as ex:
            raise(ex)

        except Exception as ex:
             self.signalling.log(3,'GasData_File.load_SpecialColumns()',f'xFAILED to load special columns \nERROR:\n{traceback.format_exc()}')
             raise(ex)

    def load_DataAndAtts(self,dataFile):
        ### read data and atts from csv

        try:

            #nb the commented out line here breaks in both QGIS 3.2 and 3.4
            #f = open(dataFile, encoding='utf-8') #utf-8 is default in QGIS 3.4 (but not 3.2)
            f = open(dataFile, encoding='Cp1252') #.gas files use THIS encoding for the internal CSV (as its still the default on windows?)

            rowCount=0
            headerRow = True
            for rawLine in csv.reader(f):
                line = (unescape(cell) for cell in rawLine)  # generator
                
                if headerRow: # skip
                    headerRow = False
                    continue
                
                tup = tuple(line)
                
                # get the columns we need and make sure numerics are numeric
                data = lookup(tup, self.columns) 

                ##self.signalling.log(2, 'GasData_File.load_data()', f'{rowCount} data: {data}')
                
                att = tup[-4:] #atts including filter (4)
                symbolicAtt = tup[-3:] #symbolic atts only (3)
                dataAndAtt = data + att
                
                #if i < 30: # debug only
                #self.signalling.log(2, 'GasData_File.load_data()', f'{rowCount} dataAndAtt {dataAndAtt}')

                #IF row is visible
                if self.colours[int(att[1])].visible \
                    and self.shapes[int(att[2])].visible \
                        and self.sizes[int(att[3])].visible:
                            if self.filters[int(att[0])].visible:
                                self.symbolicAtts.append(symbolicAtt) 
                                self.atts.append(att)
                                self.data.append(data)
                                self.dataAndAtts.append(dataAndAtt)
                                #self.signalling.log(2, 'GasData_File.load_data()', f'{rowCount} dataAndAtt {dataAndAtt}')
                
                rowCount+=1
                
            f.close()
        

        except Exception as ex:
            self.signalling.log(3,'GasData_File.load_DataAndAtts()',f'FAILED to load data from data message\nERROR:\n{traceback.format_exc()}')

class MissingCoordColumnException(Exception):
    pass

class WrongGASVersionException(Exception):
    pass